# =========================
# apps/wallets/views.py
# =========================

# ── Standard Library
import logging
from datetime import timedelta
from decimal import Decimal

# ── DRF
from rest_framework import generics, status
from rest_framework.permissions import AllowAny, IsAdminUser, IsAuthenticated
from rest_framework.response import Response
from rest_framework.views import APIView

# ── Django
from django.conf import settings
from django.contrib.auth import get_user_model
from django.db import transaction
from django.db.models import Sum
from django.shortcuts import get_object_or_404
from django.utils import timezone

# ── DRF Throttle (safe fallback)
try:
    from rest_framework.throttling import ScopedRateThrottle
except Exception:  # pragma: no cover

    class ScopedRateThrottle:  # type: ignore
        scope = None


# ── Swagger (safe fallback)
try:
    from drf_yasg import openapi
    from drf_yasg.utils import swagger_auto_schema
except Exception:  # pragma: no cover

    class _Noop:
        def __call__(self, *a, **k):
            def _inner(fn):
                return fn

            return _inner

    swagger_auto_schema = _Noop()

    class openapi:  # type: ignore
        IN_HEADER = "header"
        TYPE_STRING = "string"

        class Response(str): ...

        class Parameter:
            def __init__(self, *a, **k): ...


import secrets
from decimal import ROUND_DOWN

from eth_account import Account
from eth_account.messages import encode_defunct

# ===== Missing imports for ruff F821 =====
from rest_framework.exceptions import ValidationError
from web3 import Web3

from django.core.management import call_command
from django.db.models import Count

from apps.stakes.models import Stake
from apps.stakes.services import accrue_until_now
from apps.token_app.models import Token

# ── Project: Business logic
# ── Project: Models
from apps.wallets.models import (
    OutgoingTransaction,
    PendingReward,
    Wallet,
    WalletAuthNonce,
    WalletConnection,
    WalletTransaction,
    WithdrawalItem,
    WithdrawRequest,
)

# ── Project: Serializers
from apps.wallets.serializers import (
    WalletTransactionSerializer,
    WithdrawRequestDetailSerializer,
    WithdrawRequestListSerializer,
    WithdrawRequestSerializer,
)
from apps.wallets.service.withdrawals import process_withdraw_request

# ── Project: Services
from apps.wallets.services import claim_rewards, get_withdrawable_amount
from apps.wallets.sync_stakes import get_onchain_balance

# ── Project: Core mixins/utils
from Djangomining.feature_flags import FeatureRequiredMixin
from Djangomining.idempotency import IDEMPOTENCY_HEADER, idempotent_request

from .services import get_min_withdrawal_for_symbol

logger = logging.getLogger(__name__)

User = get_user_model()


class MetaMaskSignatureVerifyView(APIView):
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "auth"
    permission_classes = [AllowAny]

    def post(self, request):

        message = request.data.get("message")
        signature = request.data.get("signature")
        address = request.data.get("address")

        if not all([message, signature, address]):
            raise ValidationError("message, signature, and address are required")

        try:
            message_encoded = encode_defunct(text=message)
            recovered_address = Account.recover_message(message_encoded, signature=signature)
        except Exception as e:
            raise ValidationError(f"Signature verification failed: {str(e)}") from e

        if recovered_address.lower() != address.lower():
            raise ValidationError("Signature does not match address")

        return Response({"message": "Wallet verified", "address": recovered_address})


class WalletConnectView(APIView):
    permission_classes = [IsAuthenticated]

    def post(self, request):
        address = request.data.get("address")
        user_agent = request.META.get("HTTP_USER_AGENT", "").lower()

        if not address:
            raise ValidationError({"detail": "Wallet address is required"})

        # تشخیص provider از User-Agent
        if "trust" in user_agent:
            provider = "trustwallet"
        elif "safepal" in user_agent:
            provider = "safepal"
        elif "metamask" in user_agent:
            provider = "metamask"
        else:
            provider = "other"

        existing = WalletConnection.objects.filter(wallet_address=address).first()
        if existing and existing.user != request.user:
            raise ValidationError({"detail": "This wallet is already connected to another user"})

        WalletConnection.objects.filter(user=request.user).delete()

        WalletConnection.objects.create(
            user=request.user,
            wallet_address=address,
            provider=provider,
            created_at=timezone.now(),
            updated_at=timezone.now(),
        )

        return Response(
            {
                "message": f"{provider.capitalize()} wallet connected successfully",
                "wallet_address": address,
                "provider": provider,
            }
        )


class GetConnectedWalletInfoView(APIView):
    permission_classes = [IsAuthenticated]

    def get(self, request):
        provider = request.query_params.get("provider")
        qs = WalletConnection.objects.filter(user=request.user)

        if provider:
            provider = provider.lower()
            if provider not in ["metamask", "trustwallet", "safepal", "unknown"]:
                provider = "unknown"
            qs = qs.filter(provider=provider)

        connections = qs.all()
        if not connections.exists():
            return Response({"connected": False, "connections": []})

        data = []
        for conn in connections:
            wallet_address = conn.wallet_address

            if not Web3.is_address(wallet_address):
                data.append(
                    {
                        "wallet_address": wallet_address,
                        "provider": conn.provider,
                        "connected": False,
                        "error": "Invalid wallet address format",
                    }
                )
                continue

            rz_balance, mgc_balance = None, None

            try:
                rz_token = Token.objects.filter(symbol="RZ").first()
                mgc_token = Token.objects.filter(symbol="MGC").first()

                if rz_token and rz_token.contract_address and rz_token.decimals is not None:
                    rz_balance = get_onchain_balance(
                        wallet_address, rz_token.contract_address, rz_token.decimals
                    )

                if mgc_token and mgc_token.contract_address and mgc_token.decimals is not None:
                    mgc_balance = get_onchain_balance(
                        wallet_address, mgc_token.contract_address, mgc_token.decimals
                    )

            except Exception as e:
                logger.error(
                    f"[Blockchain Error] Failed to get balances for {wallet_address}: {str(e)}"
                )
                data.append(
                    {
                        "wallet_address": wallet_address,
                        "provider": conn.provider,
                        "connected": True,
                        "onchain_balances": {"RZ": None, "MGC": None},
                        "error": str(e),
                        "created_at": conn.created_at,
                        "updated_at": conn.updated_at,
                    }
                )
                continue

            data.append(
                {
                    "wallet_address": wallet_address,
                    "provider": conn.provider,
                    "connected": True,
                    "onchain_balances": {
                        "RZ": float(rz_balance) if rz_balance is not None else None,
                        "MGC": float(mgc_balance) if mgc_balance is not None else None,
                    },
                    "created_at": conn.created_at,
                    "updated_at": conn.updated_at,
                }
            )

        return Response(
            {
                "connected": True,
                "connections": data,
            }
        )


class SendPublicAddressView(APIView):
    permission_classes = [IsAuthenticated]

    def post(self, request):

        provider = request.data.get("provider", "").lower()
        wallet_address = request.data.get("wallet_address", "")

        if not wallet_address:
            raise ValidationError({"detail": "Wallet address is required"})

        if provider not in ["metamask", "trustwallet", "safepal"]:
            raise ValidationError({"detail": "Invalid wallet provider"})

        return Response(
            {
                "message": f"{provider.capitalize()} wallet address received successfully",
                "wallet_address": wallet_address,
                "provider": provider,
            }
        )


class BaseCronAPIView(APIView):
    """
    Base view for triggering cronjob management commands securely.
    """

    permission_classes = []

    command_name = None

    def get(self, request):
        print(f"CRON CALLED: {self.command_name}")  # لاگ چاپ در کنسول
        token = request.query_params.get("token")

        if token != getattr(settings, "CRONJOB_SECRET_TOKEN", None):
            logger.warning(f"Unauthorized access attempt to {self.command_name or 'unknown'} cron.")
            return Response({"detail": "Unauthorized"}, status=status.HTTP_401_UNAUTHORIZED)

        if not self.command_name:
            return Response({"detail": "No command specified."}, status=status.HTTP_400_BAD_REQUEST)

        try:
            logger.info(f"🚀 Starting cronjob: {self.command_name}")
            call_command(self.command_name)
            logger.info(f"✅ Finished cronjob: {self.command_name}")
            return Response(
                {"detail": f"✅ {self.command_name} executed successfully."}, status=200
            )
        except Exception as e:
            logger.exception(f"❌ Error running {self.command_name}: {e}")
            return Response({"detail": f"⛔ Error: {str(e)}"}, status=500)


class RunDistributeRewardsAPIView(BaseCronAPIView):
    command_name = "distribute_rewards"


class RunSyncStakeAPIView(BaseCronAPIView):
    command_name = "sync_stakes"


class MetaMaskNonceView(APIView):
    authentication_classes = []  # ← مهم: ناشناس مجاز
    permission_classes = [AllowAny]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "auth"  # یا "signup" اگر جدا کردی

    def post(self, request):
        address = (request.data.get("address") or "").strip().lower()
        if not address:
            return Response({"detail": "address is required"}, status=400)

        ttl_sec = int(getattr(settings, "WALLET_NONCE_TTL_SECONDS", 300))  # 5min default
        now = timezone.now()

        # پیام nonce
        nonce = secrets.token_hex(16)
        message = f"Login with nonce: {nonce}"

        obj = WalletAuthNonce.objects.create(
            address=address,
            provider="metamask",
            message=message,
            expires_at=now + timedelta(seconds=ttl_sec),
            is_used=False,
            user=None,
        )
        return Response(
            {
                "id": obj.id,
                "message": obj.message,
                "provider": obj.provider,
                "expires_at": obj.expires_at,
            },
            status=201,
        )


class WalletDisconnectView(APIView):
    permission_classes = [IsAuthenticated]

    def post(self, request):
        WalletConnection.objects.filter(user=request.user).delete()
        return Response({"message": "Wallet disconnected"})


class WalletConnectionStatusView(APIView):
    permission_classes = [IsAuthenticated]

    def get(self, request, *args, **kwargs):
        conns = list(
            WalletConnection.objects.filter(user=request.user)
            .order_by("-id")
            .values("provider", "wallet_address")
        )
        primary = conns[0] if conns else None

        data = {
            "connected": bool(conns),
            "primary_provider": (primary or {}).get("provider"),
            "primary_address": (primary or {}).get("wallet_address"),
            "connections": conns,  # e.g. [{"provider":"metamask","wallet_address":"0x..."}, ...]
        }
        return Response(data, status=200)


class WalletTransactionListView(generics.ListAPIView):
    serializer_class = WalletTransactionSerializer
    permission_classes = [IsAuthenticated]

    def get_queryset(self):
        return WalletTransaction.objects.filter(wallet__user=self.request.user).order_by(
            "-created_at"
        )


class WalletSummaryView(APIView):
    permission_classes = [IsAuthenticated]

    def get(self, request):
        from decimal import Decimal

        from django.db.models import Max

        from apps.wallets.models import PendingReward, WalletConnection, WalletTransaction

        user = request.user
        wallet = Wallet.objects.filter(user=user).first()

        conn = WalletConnection.objects.filter(user=user).order_by("-id").first()
        wallet_connected = bool(conn)
        wallet_address = getattr(conn, "wallet_address", None) if conn else None
        wallet_provider = getattr(conn, "provider", None) if conn else None

        # pending/claimed
        pending_rewards = PendingReward.objects.filter(user=user, status="pending").aggregate(
            total=Sum("amount")
        )["total"] or Decimal("0")

        claimed_qs = PendingReward.objects.filter(user=user, status="claimed", withdrawn=False)
        claimed_total = claimed_qs.aggregate(total=Sum("amount"))["total"] or Decimal("0")
        last_claimed_at = claimed_qs.aggregate(last=Max("claimed_at"))["last"]

        # withdrawable split
        per_token = (
            claimed_qs.values("token__symbol")
            .annotate(total=Sum("amount"))
            .order_by("token__symbol")
        )
        withdrawable_per_token = {
            row["token__symbol"]: str(row["total"]) for row in per_token if row["token__symbol"]
        }

        # txs count
        total_tx = WalletTransaction.objects.filter(wallet=wallet).count() if wallet else 0

        data = {
            "balance": str(wallet.balance) if wallet else "0",
            "pending_rewards": str(pending_rewards),
            "withdrawable_total": str(claimed_total),
            "withdrawable_per_token": withdrawable_per_token,
            "last_claimed_at": last_claimed_at.isoformat() if last_claimed_at else None,
            "wallet_connected": wallet_connected,
            "wallet_address": wallet_address,
            "wallet_provider": wallet_provider,
            "total_transactions": total_tx,
        }
        return Response(data)


class WithdrawableAmountView(APIView):
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "withdrawable"
    permission_classes = [IsAuthenticated]

    def get(self, request):
        amt = get_withdrawable_amount(request.user)
        return Response({"withdrawable": str(amt)}, status=status.HTTP_200_OK)


ADVISORY_LOCK_KEY_BASE = getattr(settings, "ADVISORY_LOCK_KEY", 4242424242)
MIN_WITHDRAWALS = getattr(settings, "MIN_WITHDRAWALS", {"MGC": Decimal("100"), "RZ": Decimal("40")})
SERVER_WALLET_ADDRESS = getattr(settings, "SERVER_WALLET_ADDRESS", None)
SERVER_WALLET_PRIVATE_KEY = getattr(settings, "SERVER_WALLET_PRIVATE_KEY", None)
BSC_CHAIN_ID = getattr(settings, "BSC_CHAIN_ID", 56)
REWARD_WAIT_RECEIPT_TIMEOUT = getattr(settings, "REWARD_WAIT_RECEIPT_TIMEOUT", 180)


def sanitize_receipt(receipt, w3):
    r = {}
    if hasattr(receipt, "transactionHash") and receipt.transactionHash is not None:
        r["transactionHash"] = w3.toHex(receipt.transactionHash)
    r["status"] = int(getattr(receipt, "status", 0) or 0)
    if getattr(receipt, "blockNumber", None) is not None:
        r["blockNumber"] = int(receipt.blockNumber)
    if getattr(receipt, "gasUsed", None) is not None:
        r["gasUsed"] = int(receipt.gasUsed)
    return r


class WithdrawCreateView(FeatureRequiredMixin, APIView):
    """
    POST /api/wallets/withdraw/
    Enqueue a withdraw request (feature-gated + idempotent).

    Flow:
      1) settle active stakes → up-to-date PendingReward
      2) validate amount/token, check withdrawable
      3) reserve PendingReward rows and create WithdrawRequest(status='queued')
      4) worker sends on-chain later (frozen for now)
    """

    required_feature = "WITHDRAW_ENABLED"
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "withdraw"
    permission_classes = [IsAuthenticated]

    def _validate_amount_precision(self, amount: Decimal, decimals: int):
        q = Decimal(1) / (Decimal(10) ** decimals)
        quantized = amount.quantize(q, rounding=ROUND_DOWN)
        if quantized != amount:
            raise ValueError(f"Amount has precision beyond token.decimals={decimals}")

    @swagger_auto_schema(
        manual_parameters=[IDEMPOTENCY_HEADER],
        operation_summary="Create withdraw request (enqueue or submit; idempotent; feature-gated)",
        responses={
            200: openapi.Response("submitted"),
            202: openapi.Response("enqueued"),
            400: "bad request",
            403: "feature disabled",
        },
    )
    @idempotent_request(cache_ttl=60)  # جلوگیری از دوباره‌کاری ناخواسته
    def post(self, request, *args, **kwargs):
        user = request.user
        ser = WithdrawRequestSerializer(data=request.data)
        ser.is_valid(raise_exception=True)

        amount: Decimal = ser.validated_data["amount"]
        token_symbol: str = ser.validated_data.get("token", None)

        if amount <= 0:
            return Response({"detail": "Amount must be > 0"}, status=400)

        # اگر توکن داده شد: دقت و حداقل را چک کن
        token = None
        token_decimals = 18
        if token_symbol:
            token = get_object_or_404(Token, symbol__iexact=token_symbol)
            token_decimals = int(getattr(token, "decimals", 18))

            # دقت عدد
            try:
                self._validate_amount_precision(amount, token_decimals)
            except ValueError as e:
                return Response({"detail": str(e)}, status=400)

            # حداقل برداشت
            min_allowed = get_min_withdrawal_for_symbol(token_symbol)
            if amount < min_allowed:
                return Response(
                    {"detail": f"Minimum withdraw for {token_symbol.upper()} is {min_allowed}"},
                    status=400,
                )

        # 1) قبل از برداشت، stakeهای فعال را settle کن تا pending به‌روز شوند
        with transaction.atomic():
            active_stakes = Stake.objects.select_for_update().filter(user=user, is_active=True)
            for st in active_stakes:
                accrue_until_now(st)

        # 2) موجودی withdrawable تازه‌محاسبه‌شده
        total_withdrawable = get_withdrawable_amount(user)
        if amount > total_withdrawable:
            return Response(
                {
                    "detail": "Requested amount exceeds withdrawable balance",
                    "withdrawable_total": str(total_withdrawable),
                },
                status=400,
            )

        # 3) گزینش PendingRewardهای قابل مصرف (claimed & not withdrawn)
        pr_qs = (
            PendingReward.objects.select_for_update()
            .filter(user=user, status="claimed", withdrawn=False)
            .order_by("id")
        )
        if token:
            pr_qs = pr_qs.filter(token=token)

        gathered = []
        acc = Decimal("0")

        with transaction.atomic():
            # جمع تا رسیدن به amount
            for pr in pr_qs:
                gathered.append(pr)
                acc += pr.amount
                if acc >= amount:
                    break

            if acc < amount:
                return Response(
                    {
                        "detail": (
                            "Insufficient withdrawable in selected token"
                            if token
                            else "Insufficient withdrawable"
                        )
                    },
                    status=400,
                )

            # 4) ساخت درخواست برداشت
            wr = WithdrawRequest.objects.create(
                user=user,
                amount=amount,
                status="queued",  # در صف تا زمانی که worker/سرویس ارسال کند
            )

            # 5) مصرف منطقی PendingRewardها به اندازه amount
            amount_left = amount
            for pr in gathered:
                if amount_left <= 0:
                    break

                if pr.amount <= amount_left:
                    consumed = pr.amount
                    pr.withdrawn = True  # مصرف شد برای برداشت
                    pr.save(update_fields=["withdrawn"])
                    # اگر WithdrawalItem فیلد FK به WithdrawRequest دارد، ست کن
                    wi_kwargs = {
                        "pending_reward": pr,
                        "outgoing_tx": None,
                        "consumed_amount": consumed,
                    }
                    if "withdrawrequest" in [f.name for f in WithdrawalItem._meta.get_fields()]:
                        wi_kwargs["withdrawrequest"] = wr
                    WithdrawalItem.objects.create(**wi_kwargs)
                    amount_left -= consumed
                else:
                    consumed = amount_left
                    pr.amount = pr.amount - consumed
                    pr.save(update_fields=["amount"])
                    wi_kwargs = {
                        "pending_reward": pr,
                        "outgoing_tx": None,
                        "consumed_amount": consumed,
                    }
                    if "withdrawrequest" in [f.name for f in WithdrawalItem._meta.get_fields()]:
                        wi_kwargs["withdrawrequest"] = wr
                    WithdrawalItem.objects.create(**wi_kwargs)
                    amount_left = Decimal("0")

        # 6) بسته به تنظیمات: همین‌جا submit کن یا در صف بماند
        if getattr(settings, "WITHDRAW_SYNC_SEND", False):
            # ارسال فوری (فعلاً mock در service.withdrawals)
            try:
                tx = process_withdraw_request(wr)
            except Exception as e:
                # اگر مشکلی پیش آمد، درخواست در حالت queued می‌ماند
                return Response(
                    {
                        "detail": "enqueued",
                        "request_id": wr.id,
                        "amount": str(wr.amount),
                        "token": (token.symbol if token else None),
                        "queued_at": timezone.now().isoformat(),
                        "note": f"submit failed -> queued ({e})",
                    },
                    status=status.HTTP_202_ACCEPTED,
                )

            return Response(
                {
                    "detail": "submitted",
                    "request_id": wr.id,
                    "amount": str(wr.amount),
                    "token": (token.symbol if token else None),
                    "tx_id": str(tx.id),
                    "tx_hash": tx.tx_hash,
                    "status": tx.status,
                    "submitted_at": timezone.now().isoformat(),
                },
                status=status.HTTP_200_OK,
            )

        # حالت صف (پیش‌فرض امن در dev)
        return Response(
            {
                "detail": "enqueued",
                "request_id": wr.id,
                "amount": str(wr.amount),
                "token": (token.symbol if token else None),
                "queued_at": timezone.now().isoformat(),
            },
            status=status.HTTP_202_ACCEPTED,
        )


class WithdrawStatusView(FeatureRequiredMixin, APIView):
    """
    GET /api/wallets/withdraw/status/?limit&offset
    Paginated view of user's outgoing on-chain txs or queued requests.
    """

    permission_classes = [IsAuthenticated]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "withdraw"
    feature_name = "WITHDRAW_ENABLED"

    @swagger_auto_schema(
        manual_parameters=[
            openapi.Parameter(
                "id",
                openapi.IN_QUERY,
                type=openapi.TYPE_INTEGER,
                required=False,
                description="WithdrawRequest ID (اختیاری، اگر بدهید وضعیت همان یکی را می‌گیرید)",
            ),
            openapi.Parameter(
                "limit",
                openapi.IN_QUERY,
                type=openapi.TYPE_INTEGER,
                required=False,
                default=10,
                description="حداکثر تعداد ردیف در لیست (بدون id)",
            ),
            openapi.Parameter(
                "offset",
                openapi.IN_QUERY,
                type=openapi.TYPE_INTEGER,
                required=False,
                default=0,
                description="جابه‌جایی (بدون id)",
            ),
        ],
        operation_summary="Get withdraw status (single by id, or recent list)",
        responses={200: "ok", 400: "bad request", 404: "not found"},
    )
    def get(self, request):
        from apps.wallets.models import (  # اطمینان از import محلی
            OutgoingTransaction,
            WithdrawRequest,
        )

        # اگر id داده شده: وضعیت یک برداشت را برگردان
        raw_id = request.query_params.get("id")
        if raw_id is not None:
            try:
                wr_id = int(raw_id)
            except ValueError:
                return Response({"detail": "id must be an integer"}, status=400)

            # اولویت: WithdrawRequest (فلو جدید/ماک)
            wr = WithdrawRequest.objects.filter(id=wr_id, user=request.user).first()
            if wr:
                payload = {
                    "kind": "withdraw_request",
                    "id": wr.id,
                    "amount": str(wr.amount),
                    "status": wr.status,
                    "approved_at": getattr(wr, "approved_at", None),
                    "completed_at": getattr(wr, "completed_at", None),
                    "created_at": (
                        getattr(wr, "created_at", None).isoformat()
                        if getattr(wr, "created_at", None)
                        else None
                    ),
                    "updated_at": (
                        getattr(wr, "updated_at", None).isoformat()
                        if getattr(wr, "updated_at", None)
                        else None
                    ),
                }
                return Response(payload, status=200)

            # اگر WR نبود، شاید در منطق قدیمی OutgoingTransaction ثبت شده باشد
            otx = OutgoingTransaction.objects.filter(id=wr_id, user=request.user).first()
            if otx:
                payload = {
                    "kind": "outgoing_tx",
                    "id": otx.id,
                    "token": getattr(getattr(otx, "token", None), "symbol", None),
                    "amount": str(otx.amount),
                    "tx_hash": getattr(otx, "tx_hash", None),
                    "status": otx.status,
                    "created_at": (
                        otx.created_at.isoformat() if getattr(otx, "created_at", None) else None
                    ),
                    "updated_at": (
                        otx.updated_at.isoformat() if getattr(otx, "updated_at", None) else None
                    ),
                    "details": getattr(otx, "details", {}) or {},
                }
                return Response(payload, status=200)

            return Response({"detail": "not found"}, status=404)

        # در غیر این صورت: لیست اخیر (لیست ترکیبی از WR جدید و OTX قدیمی)
        # پارامترها
        try:
            limit = max(1, min(50, int(request.query_params.get("limit", 10))))
            offset = max(0, int(request.query_params.get("offset", 0)))
        except ValueError:
            return Response({"detail": "limit/offset must be integers"}, status=400)

        # دو منبع را می‌گیریم و merge زمانی می‌کنیم (created_at/updated_at)
        wr_qs = WithdrawRequest.objects.filter(user=request.user).order_by(
            "-id"
        )  # اگر created_at داری، بهتره بر اساس created_at desc

        otx_qs = OutgoingTransaction.objects.filter(user=request.user).order_by("-id")

        # به شکل ساده: فقط WRها را اولویت بده؛ اگر خواستی ترکیبی کن
        # برای سادگی فعلاً: فقط WR (فلو جدید)، و اگر چیزی نبود از قدیمی‌ها
        items = []

        for wr in wr_qs[offset : offset + limit]:
            items.append(
                {
                    "kind": "withdraw_request",
                    "id": wr.id,
                    "amount": str(wr.amount),
                    "status": wr.status,
                    "approved_at": getattr(wr, "approved_at", None),
                    "completed_at": getattr(wr, "completed_at", None),
                    "created_at": (
                        getattr(wr, "created_at", None).isoformat()
                        if getattr(wr, "created_at", None)
                        else None
                    ),
                    "updated_at": (
                        getattr(wr, "updated_at", None).isoformat()
                        if getattr(wr, "updated_at", None)
                        else None
                    ),
                }
            )

        # اگر WR خالی بود، برای سازگاری عقب‌رو به OTX برگردیم
        if not items:
            for tx in otx_qs[offset : offset + limit]:
                items.append(
                    {
                        "kind": "outgoing_tx",
                        "id": tx.id,
                        "token": getattr(getattr(tx, "token", None), "symbol", None),
                        "amount": str(tx.amount),
                        "tx_hash": getattr(tx, "tx_hash", None),
                        "status": tx.status,
                        "created_at": (
                            tx.created_at.isoformat() if getattr(tx, "created_at", None) else None
                        ),
                        "updated_at": (
                            tx.updated_at.isoformat() if getattr(tx, "updated_at", None) else None
                        ),
                        "details": getattr(tx, "details", {}) or {},
                    }
                )

        return Response(
            {
                "count": len(items),
                "results": items,
                "limit": limit,
                "offset": offset,
            },
            status=200,
        )


class RecentTransactionsView(APIView):
    permission_classes = [IsAuthenticated]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "wallet"  # اگر scopes داری؛ در settings نرخشو داری/اضافه کن

    def get(self, request, *args, **kwargs):
        try:
            limit = int(request.query_params.get("limit", 20))
        except ValueError:
            limit = 20
        limit = max(1, min(limit, 50))  # امنیت

        qs = (
            OutgoingTransaction.objects.filter(user=request.user)
            .select_related("token")
            .order_by("-created_at")[:limit]
        )

        items = []
        for tx in qs:
            items.append(
                {
                    "id": tx.id,
                    "type": "withdraw",
                    "token": getattr(tx.token, "symbol", None),
                    "amount": str(tx.amount),
                    "status": tx.status,  # e.g. sending/sent/failed
                    "tx_hash": getattr(tx, "tx_hash", None),
                    "created_at": tx.created_at.isoformat(),
                }
            )
        return Response({"results": items}, status=status.HTTP_200_OK)


class ClaimRewardsView(FeatureRequiredMixin, APIView):
    """
    POST /api/wallets/claim/
    Idempotent claim endpoint: settles active stakes up to now, then moves
    user's PENDING rewards into CLAIMED bucket atomically.
    """

    permission_classes = [IsAuthenticated]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "claim"
    feature_name = "CLAIM_ENABLED"  # ← قابلیت قابل خاموش/روشن شدن

    @swagger_auto_schema(
        manual_parameters=[IDEMPOTENCY_HEADER],
        operation_summary="Claim rewards (idempotent)",
        responses={200: openapi.Response("claimed")},
    )
    @idempotent_request(cache_ttl=60)  # درخواست‌های تکراری 60ثانیه‌ای را idempotent می‌کند
    def post(self, request):
        from apps.stakes.models import Stake
        from apps.stakes.services import accrue_until_now  # تسویهٔ سود تا اکنون

        user = request.user

        # 1) قبل از claim، سود استیک‌های فعال را تا اکنون settle کن (در بلاک اتمیک با قفل ردیفی)
        try:
            with transaction.atomic():
                active_qs = (
                    Stake.objects.select_for_update()
                    .filter(user=user, is_active=True)
                    .order_by("id")
                )
                for st in active_qs:
                    # اگر accrue_until_now خودش اتمیک بود، مشکلی نیست؛ در همین تراکنش فراخوانی می‌شود.
                    accrue_until_now(st)
        except Exception as e:
            return Response(
                {"detail": f"settle_error: {e}"},
                status=status.HTTP_400_BAD_REQUEST,
            )

        # 2) حالا claim انجام بده (pending → claimed) و جمع‌ها را بگیر
        try:
            result = claim_rewards(user)
        except Exception as e:
            return Response(
                {"detail": f"claim_error: {e}"},
                status=status.HTTP_400_BAD_REQUEST,
            )

        # 3) اگر چیزی برای claim نبود
        count = int(result.get("count", 0) or 0)
        if count == 0:
            return Response(
                {"detail": "Nothing to claim.", "claimed_total": "0"},
                status=status.HTTP_200_OK,
            )

        # 4) سریال‌سازی خروجی (Decimal → str/float امن)
        claimed_total = result.get("claimed_total", Decimal("0"))
        try:
            claimed_total = Decimal(claimed_total)
        except Exception:
            claimed_total = Decimal("0")

        by_token_raw = result.get(
            "by_token", []
        )  # انتظار: iterable از dict با کلیدهای symbol / total
        by_token = []
        for row in by_token_raw:
            sym = row.get("symbol")
            tot = row.get("total", Decimal("0"))
            try:
                tot = Decimal(tot)
            except Exception:
                tot = Decimal("0")
            by_token.append(
                {
                    "symbol": sym,
                    "total": float(tot),  # برای راحتی فرانت
                }
            )

        return Response(
            {
                "detail": "claimed",
                "claimed_total": float(claimed_total),
                "count": count,
                "by_token": by_token,
                "claimed_at": result.get("claimed_at") or timezone.now().isoformat(),
            },
            status=status.HTTP_200_OK,
        )


class WithdrawHistoryView(APIView):
    """
    GET /api/wallets/withdraw/history
    لیست آخرین درخواست‌های برداشت کاربر (paged سبک: فعلاً 20 تا).
    """

    permission_classes = [IsAuthenticated]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "withdraw"

    @swagger_auto_schema(
        operation_summary="List my recent withdrawal requests",
        responses={200: openapi.Response("OK", WithdrawRequestListSerializer(many=True))},
    )
    def get(self, request):
        qs = (
            WithdrawRequest.objects.filter(user=request.user)
            .order_by("-id")
            .annotate(items_count=Count("withdrawalitem"))
        )[:20]

        data = []
        for wr in qs:
            # tx_hash را اگر مستقیم روی مدل هست بگیر
            tx_hash = None
            for cand in ("tx_hash", "txid", "transaction_hash"):
                if hasattr(wr, cand) and getattr(wr, cand):
                    tx_hash = getattr(wr, cand)
                    break
            # یا از آیتم‌ها (اگر خروجی‌تراکنش دارد)
            if tx_hash is None:
                try:
                    first_item = (
                        WithdrawalItem.objects.filter(
                            outgoing_tx__isnull=False,
                            outgoing_tx__tx_hash__isnull=False,
                            outgoing_tx__id__isnull=False,
                            outgoing_tx__withdrawrequest=wr,
                        )
                        .order_by("id")
                        .first()
                    )  # اگر چنین رابطه‌ای داری
                except Exception:
                    first_item = None
                if (
                    first_item
                    and hasattr(first_item, "outgoing_tx")
                    and getattr(first_item.outgoing_tx, "tx_hash", None)
                ):
                    tx_hash = first_item.outgoing_tx.tx_hash

            data.append(
                {
                    "id": wr.id,
                    "amount": str(getattr(wr, "amount", "")),
                    "status": getattr(wr, "status", ""),
                    "created_at": getattr(wr, "created_at", None),
                    "updated_at": getattr(wr, "updated_at", None),
                    "tx_hash": tx_hash,
                    "items_count": getattr(wr, "items_count", 0),
                }
            )
        ser = WithdrawRequestListSerializer(data, many=True)
        return Response(ser.data, status=200)


class WithdrawStatusDetailView(APIView):
    """
    GET /api/wallets/withdraw/<int:pk>/status
    جزئیات یک برداشت (به همراه آیتم‌ها).
    """

    permission_classes = [IsAuthenticated]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "withdraw"

    @swagger_auto_schema(
        operation_summary="Withdrawal request detail/status",
        responses={200: openapi.Response("OK", WithdrawRequestDetailSerializer)},
    )
    def get(self, request, pk):
        wr = get_object_or_404(WithdrawRequest, pk=pk, user=request.user)

        # tx_hash (مثل بالا)
        tx_hash = None
        for cand in ("tx_hash", "txid", "transaction_hash"):
            if hasattr(wr, cand) and getattr(wr, cand):
                tx_hash = getattr(wr, cand)
                break

        # آیتم‌ها — اگر WithdrawalItem دارید
        items_payload = []
        try:
            items_qs = WithdrawalItem.objects.filter(
                outgoing_tx__withdrawrequest=wr
            ).select_related("pending_reward", "outgoing_tx")
        except Exception:
            # رابطه ممکن است متفاوت باشد: بگردیم با کلیدهای معمول
            try:
                items_qs = WithdrawalItem.objects.filter(outgoing_tx__isnull=False).order_by("id")
            except Exception:
                items_qs = []

        for it in items_qs:
            pr = getattr(it, "pending_reward", None)
            tok = getattr(pr, "token", None)
            sym = getattr(tok, "symbol", None)
            items_payload.append(
                {
                    "id": it.id,
                    "consumed_amount": str(getattr(it, "consumed_amount", "")),
                    "token": sym,
                    "pending_reward_id": getattr(pr, "id", None) if pr else None,
                    "outgoing_tx_id": getattr(getattr(it, "outgoing_tx", None), "id", None),
                }
            )

        payload = {
            "id": wr.id,
            "amount": str(getattr(wr, "amount", "")),
            "status": getattr(wr, "status", ""),
            "created_at": getattr(wr, "created_at", None),
            "updated_at": getattr(wr, "updated_at", None),
            "tx_hash": tx_hash,
            "items": items_payload,
        }
        ser = WithdrawRequestDetailSerializer(payload)
        return Response(ser.data, status=200)


# --- Admin-only: approve withdraw ---
class WithdrawApproveView(APIView):
    permission_classes = [IsAdminUser]

    @swagger_auto_schema(
        operation_summary="Admin: approve a WithdrawRequest",
        request_body=openapi.Schema(
            type=openapi.TYPE_OBJECT,
            properties={"request_id": openapi.Schema(type=openapi.TYPE_INTEGER)},
            required=["request_id"],
        ),
        responses={200: "ok", 404: "not found", 400: "bad req"},
    )
    @transaction.atomic
    def post(self, request, *args, **kwargs):
        rid = request.data.get("request_id")
        wr = get_object_or_404(WithdrawRequest, pk=rid)
        if wr.status != "pending":
            return Response({"detail": f"invalid state: {wr.status}"}, status=400)
        wr.status = "approved"
        wr.save(update_fields=["status", "updated_at"])
        return Response({"detail": "approved", "id": wr.id})


class WithdrawMockProcessView(APIView):
    """
    POST /api/wallets/withdraw/mock_process/
    body: {"id": <WithdrawRequest.id>}
    فقط برای لوکال/ادمین: همان کار management command را برای یک آی‌دی انجام می‌دهد.
    """

    permission_classes = [IsAdminUser]

    @swagger_auto_schema(
        operation_summary="(ADMIN) Mock process a single withdraw by ID",
        request_body=openapi.Schema(
            type=openapi.TYPE_OBJECT,
            properties={"id": openapi.Schema(type=openapi.TYPE_INTEGER)},
            required=["id"],
        ),
        responses={200: "processed", 400: "bad request"},
    )
    def post(self, request):
        wr_id = request.data.get("id")
        if not wr_id:
            return Response({"detail": "id is required"}, status=400)
        try:
            call_command("mock_process_withdraws", id=int(wr_id), limit=1)
        except Exception as e:
            return Response({"detail": str(e)}, status=400)
        return Response({"detail": "processed", "id": wr_id}, status=200)
