from __future__ import annotations

import uuid

from django.db import transaction
from django.utils import timezone

from apps.wallets.constants import SERVER_WALLET_ADDRESS
from apps.wallets.models import (
    NonceReservation,
    OutgoingTransaction,
    WithdrawalItem,
    WithdrawRequest,
)

"""
Withdrawals service.

This module encapsulates withdraw business logic (enqueue/consume PendingReward),
separate from API views:

- process_withdraw_request(wr_id) -> None (to be called by a worker when un-frozen)

Guidelines:
- Keep DB transactions short and deterministic.
- Do not call out to web3/providers inside long transactions.
"""


def _infer_token_for_wr(wr: WithdrawRequest):
    """
    از روی WithdrawalItem -> PendingReward -> token،
    توکن مربوط به این درخواست برداشت را حدس می‌زند.
    """
    item = (
        WithdrawalItem.objects.filter(withdrawrequest=wr)
        .select_related("pending_reward__token")
        .first()
    )
    if item and item.pending_reward and item.pending_reward.token_id:
        return item.pending_reward.token
    return None


def _reserve_nonce(server_address: str) -> int:
    """
    رزرو nonce برای آدرس سرور. (الآن mock: از 1 شروع می‌کنیم)
    وقتی Web3 وصل شد، اینجا اول روی‌چین nonce می‌خوانیم.
    """
    with transaction.atomic():
        nr, created = NonceReservation.objects.select_for_update().get_or_create(
            address=server_address
        )
        if created or nr.next_nonce in (None, 0):
            nr.next_nonce = 1  # TODO: بعداً از chain واقعی بخوان
        reserved = nr.next_nonce
        nr.next_nonce = nr.next_nonce + 1
        nr.save(update_fields=["next_nonce", "updated_at"])
        return int(reserved)


@transaction.atomic
def process_withdraw_request(wr: WithdrawRequest) -> OutgoingTransaction:
    """
    Worker entrypoint — send funds on-chain and finalize accounting.

    NOTE: This is currently **frozen** until manager provides final chain spec/flow.
    Keep signature stable; implement when unblocked.
    """
    if wr.status not in ("pending", "queued"):
        raise ValueError("WithdrawRequest is not pending/queued.")

    # 1) nonce رزرو
    nonce = _reserve_nonce(SERVER_WALLET_ADDRESS)

    # 2) تشخیص توکن از آیتم‌ها
    token = _infer_token_for_wr(wr)

    # 3) ساخت تراکنش خروجی (فعلاً mock)
    tx = OutgoingTransaction.objects.create(
        user=wr.user,
        token=token,
        token_contract=getattr(token, "contract_address", None),
        amount=wr.amount,
        destination_address=None,  # اگر مقصد جایی ذخیره می‌کنی اینجا ست کن
        tx_hash=f"0x{uuid.uuid4().hex}",  # MOCK تا وقتی web3 وصل شه
        status="submitted",
        details={"nonce": nonce, "mock": True},
    )

    # 4) وصل کردن WithdrawalItemها به این تراکنش
    WithdrawalItem.objects.filter(withdrawrequest=wr).update(outgoing_tx=tx)

    # 5) آپدیت وضعیت درخواست
    wr.status = "submitted"
    wr.updated_at = timezone.now()
    wr.save(update_fields=["status", "updated_at"])

    return tx
