# management/commands/update_daily_earnings.py

from decimal import Decimal

from django.core.management.base import BaseCommand
from django.db import transaction
from django.utils import timezone

from apps.miners.models import UserMiner
from apps.reward.models import Reward
from apps.users.services.referrals import compute_final_percent


class Command(BaseCommand):
    help = "Calculates and creates daily reward records for all active miners."

    @transaction.atomic
    def handle(self, *args, **options):
        today = timezone.now().date()
        active_stakes = UserMiner.objects.filter(
            is_online=True, staked_amount__gt=0
        ).select_related("user", "miner", "miner__plan")

        if not active_stakes.exists():
            self.stdout.write(
                self.style.WARNING("هیچ استیک فعالی برای محاسبه سود روزانه پیدا نشد.")
            )
            return

        rewards_to_create = []
        for stake in active_stakes:

            already_exists = Reward.objects.filter(
                user=stake.user, miner=stake.miner, created_at__date=today
            ).exists()

            if already_exists:
                continue

            # محاسبه سود روزانه
            plan = stake.miner.plan
            if not plan or not plan.monthly_reward_percent:
                continue

            staked_amount = stake.staked_amount
            monthly_percent_base = plan.monthly_reward_percent or Decimal("4.5")

            # ✅ اعمال بونس ریفرال روی درصد ماهانه
            user = stake.user if hasattr(stake, "user") else None
            monthly_percent_effective = (
                compute_final_percent(user, monthly_percent_base) if user else monthly_percent_base
            )

            # محاسبه با درصد مؤثر
            daily_reward = (
                (staked_amount * monthly_percent_effective) / Decimal("100") / Decimal("30")
            )

            if daily_reward > 0:
                rewards_to_create.append(
                    Reward(
                        user=stake.user,
                        miner=stake.miner,
                        amount=daily_reward,
                        status="paid",  # این "paid" یعنی سود این روز "واریز" شده به حساب داخلی کاربر
                    )
                )

        if rewards_to_create:
            Reward.objects.bulk_create(rewards_to_create)
            self.stdout.write(
                self.style.SUCCESS(
                    f"✅ {len(rewards_to_create)} رکورد سود روزانه با موفقیت ثبت شد."
                )
            )
        else:
            self.stdout.write(
                self.style.WARNING("برای استیک‌های فعال، سود امروز قبلاً محاسبه شده است.")
            )
