# apps/wallets/management/commands/mock_process_withdraws.py

from django.core.management.base import BaseCommand
from django.db import transaction
from django.utils import timezone

from apps.wallets.models import WalletTransaction, WithdrawRequest


class Command(BaseCommand):
    help = "Mock processor: approve & complete pending withdraw requests (no on-chain)."

    def add_arguments(self, parser):
        parser.add_argument(
            "--id", type=int, help="Process only this WithdrawRequest ID", default=None
        )
        parser.add_argument("--limit", type=int, help="Max items to process", default=50)

    def handle(self, *args, **opts):
        wr_qs = WithdrawRequest.objects.filter(status__in=["pending", "queued"]).order_by("id")
        if opts["id"]:
            wr_qs = wr_qs.filter(id=opts["id"])
        wr_qs = wr_qs[: opts["limit"]]

        if not wr_qs.exists():
            self.stdout.write(self.style.WARNING("No withdraws to process."))
            return

        count = 0
        for wr in wr_qs:
            with transaction.atomic():
                # مرحلهٔ تایید (مدیر/سیستم)
                wr.status = "approved"
                wr.approved_at = timezone.now() if hasattr(wr, "approved_at") else None
                wr.save(
                    update_fields=["status"]
                    + (["approved_at"] if hasattr(wr, "approved_at") else [])
                )

                # ایجاد یک تراکنش نمایشی (DB only)
                if hasattr(WalletTransaction, "tx_hash"):
                    tx_hash = f"MOCK-{wr.id}-{int(timezone.now().timestamp())}"
                else:
                    tx_hash = None

                # اگر WalletTransaction شما به wallet فیلد ForeignKey دارد، بهترین کار:
                wallet = getattr(getattr(wr, "user", None), "wallet", None)
                if wallet is None:
                    wallet = None

                WalletTransaction.objects.create(
                    wallet=wallet,
                    amount=wr.amount,
                    tx_type="withdraw",
                    status="done",
                    tx_hash=tx_hash,
                    created_at=timezone.now() if hasattr(WalletTransaction, "created_at") else None,
                )

                # تکمیل برداشت
                wr.status = "completed"
                wr.completed_at = timezone.now() if hasattr(wr, "completed_at") else None
                wr.save(
                    update_fields=["status"]
                    + (["completed_at"] if hasattr(wr, "completed_at") else [])
                )

                count += 1

        self.stdout.write(self.style.SUCCESS(f"Processed {count} withdraw(s) in mock mode."))
