from decimal import Decimal

from django.conf import settings
from django.db import models
from django.utils import timezone

from apps.miners.models import Miner
from apps.token_app.models import Token


class Stake(models.Model):
    class Meta:
        ordering = ["-created_at"]
        verbose_name = "Stake"
        verbose_name_plural = "Stakes"
        constraints = [
            models.UniqueConstraint(
                fields=["user", "token"],
                condition=models.Q(is_active=True),
                name="uniq_active_stake_per_user_token",
            )
        ]

    def __str__(self):
        return f"Stake #{self.id} by {self.user.email}"

    user = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name="stakes",
        verbose_name="User",
    )
    miner = models.ForeignKey(
        Miner, on_delete=models.CASCADE, related_name="stakes", verbose_name="Miner"
    )
    token = models.ForeignKey(
        Token, on_delete=models.CASCADE, verbose_name="Token", related_name="stakes"
    )
    amount = models.DecimalField(max_digits=30, decimal_places=8, verbose_name="Amount")
    created_at = models.DateTimeField(auto_now_add=True, verbose_name="Created At")
    is_active = models.BooleanField(default=True)  # ← جدید: وضعیت فعال بودن این استیک/ماینر
    last_accrual_at = models.DateTimeField(null=True, blank=True)
    # --- جدید برای محاسبهٔ لحظه‌ای:
    last_calculated_at = models.DateTimeField(null=True, blank=True)  # آخرین لحظهٔ تسویهٔ داخلی
    pending_reward = models.DecimalField(
        max_digits=30, decimal_places=8, default=Decimal("0")
    )  # جمع تا لحظهٔ آخر
    last_rate_hash = models.CharField(
        max_length=64, null=True, blank=True
    )  # برای تشخیص تغییر نرخ مؤثر

    auto_paused = models.BooleanField(default=False)  # ← جدید: توقف خودکار بر اساس بالانس
    auto_pause_reason = models.CharField(max_length=64, blank=True, default="")  # e.g. "insufficient_balance"
    last_balance_check_at = models.DateTimeField(null=True, blank=True)  # زمان آخرین چک
    paused_at = models.DateTimeField(null=True, blank=True)

    def touch_accumulator(self, now=None):
        """
        آپدیت کوچک: اگر first-time است، فقط last_calculated_at را ست می‌کند.
        این متد را بعداً در create stake یا activate stake صدا می‌زنیم.
        """
        if now is None:
            now = timezone.now()
        if self.last_calculated_at is None:
            self.last_calculated_at = now
            self.save(update_fields=["last_calculated_at"])



