from __future__ import annotations

from datetime import timedelta, datetime, time
from decimal import Decimal
from .models import UserDailyLogin
from django.contrib.auth import get_user_model
from django.contrib.contenttypes.models import ContentType
from django.db import transaction
from django.utils import timezone
from decimal import Decimal
from django.db import transaction
from django.utils import timezone

from apps.claims.models import ClaimIntent, ClaimPurpose
from apps.stakes.models import Stake
from apps.token_app.models import Token
from apps.miners.models import UserMiner
from .models import Task, UserTaskProgress, TaskConditionCode, ProgressStatus

from apps.miners.models import UserMiner
from apps.stakes.models import Stake
from apps.token_app.models import Token
from decimal import Decimal
from django.db import transaction
from apps.events.models import Task, UserTaskProgress, TaskConditionCode, ProgressStatus


from django.contrib.auth import get_user_model

from apps.miners.models import UserMiner
from apps.wallets.models import WalletConnection
from apps.claims.models import ClaimTx, ClaimPurpose
from apps.claims.services import make_claim_signature  # existing tx builder for claims
from apps.claims.services import get_pending_by_symbol_for_user


from .constants import RewardType, TaskConditionCode, TaskKind
from .models import (
    EnergyTurboDefinition,
    MinerAccessDefinition,
    ProgressStatus,
    Task,
    UserEnergyBoost,
    UserTaskProgress,
    UserTemporaryMinerAccess,
    UserDailyLogin,
)
from .constants import TaskConditionCode

User = get_user_model()
@transaction.atomic
def evaluate_and_reward_for_user(user):
    """
    همه تسک‌هایی که COMPLETE شده‌اند و هنوز reward نگرفته‌اند را پیدا کن،
    reward بده، و status را به 'rewarded' تغییر بده.
    """
    now = timezone.now()
    progresses = UserTaskProgress.objects.filter(
        user=user,
        status=ProgressStatus.COMPLETED,
        rewarded_at__isnull=True,
    ).select_related("task")

    for prg in progresses:
        task = prg.task
        if task.reward_type == TaskRewardType.ENERGY_TURBO:
            _activate_energy_turbo(user, task, now=now)
        elif task.reward_type == TaskRewardType.MINER_ACCESS:
            _activate_miner_access(user, task, now=now)

        prg.rewarded_at = now
        prg.status = ProgressStatus.REWARDED
        prg.save(update_fields=["rewarded_at", "status"])
def _date_in_window(date, window: tuple) -> bool:
    start, end = window
    if not start or not end:
        return False

    if isinstance(start, datetime):
        start = start.date()
    if isinstance(end, datetime):
        end = end.date()

    return start <= date < end

def get_pending_rewards_for_symbol(user: User, symbol: str) -> Decimal:
    """
    Get total pending rewards for user and token symbol.
    Uses existing reward utility.
    """
    pending = get_pending_by_symbol_for_user(user, symbol)
    return Decimal(pending or 0)


@transaction.atomic
def apply_restake_after_claim(
    *,
    user_address: str,
    token_address: str,
    amount_scaled: str | int,
    nonce: int,
    tx_hash: str,
) -> None:
    """
    این تابع بعد از تأیید تراکنش روی chain (از طریق وب‌هوک QuickNode) صدا زده می‌شود.

    اگر ClaimIntent مربوطه از نوع RESTAKE باشد:
      - مقدار restake شده را به Stake کاربر اضافه می‌کنیم
      - در صورت داشتن target_miner، staked_amount ماینر را هم زیاد می‌کنیم
      - روی intent.meta علامت restake_applied می‌زنیم تا idempotent باشد
      - تسک weekly restake را کامل می‌کنیم و evaluate_and_reward_for_user را صدا می‌زنیم
    """

    # 1) پیدا کردن intent مربوط به این nonce و آدرس
    intent = (
        ClaimIntent.objects.select_for_update()
        .filter(
            user_address__iexact=user_address,
            nonce=nonce,
        )
        .order_by("-id")
        .first()
    )
    if not intent:
        return

    # فقط برای restake
    if intent.purpose != ClaimPurpose.RESTAKE:
        return

    meta = intent.meta or {}
    # اگر قبلاً اعمال شده، تکرار نکن
    if meta.get("restake_applied"):
        return

    user = intent.user
    if not user:
        return

    # 2) پیدا کردن Token مرتبط
    token = None

    # اول با آدرس قرارداد (هم contract_address هم token_address را چک می‌کنیم)
    if intent.token_address:
        token = (
            Token.objects.filter(contract_address__iexact=intent.token_address).first()
            or Token.objects.filter(token_address__iexact=intent.token_address).first()
        )

    # اگر هنوز پیدا نشد، با symbol امتحان کن
    if token is None and getattr(intent, "token_symbol", ""):
        token = Token.objects.filter(symbol__iexact=intent.token_symbol).first()

    if token is None:
        # نمی‌خواهیم سیستم بترکه، فقط restake را اعمال نمی‌کنیم
        return

    decimals = int(getattr(token, "decimals", 18) or 18)
    amount_dec = Decimal(str(amount_scaled)) / (Decimal(10) ** Decimal(decimals))
    if amount_dec <= 0:
        return

    # 3) پیدا کردن یه Stake برای همین توکن
    stake = (
        Stake.objects.select_for_update()
        .filter(user=user, token=token)
        .order_by("-id")
        .first()
    )

    # اگر stake نداریم، می‌تونی بسازی یا سایلنت رد شوی؛ این‌جا می‌سازیم
    if stake is None:
        stake = Stake.objects.create(
            user=user,
            token=token,
            amount=Decimal("0"),
        )

    stake.amount = (Decimal(stake.amount) + amount_dec).quantize(Decimal("0.00000001"))
    stake.save(update_fields=["amount"])

    # 4) اگر target_miner برای intent ست شده، staked_amount ماینر را هم زیاد کن
    target_um: UserMiner | None = intent.target_miner
    if target_um is not None:
        target_um.staked_amount = (
            Decimal(target_um.staked_amount) + amount_dec
        ).quantize(Decimal("0.00000001"))
        target_um.save(update_fields=["staked_amount"])

    # 5) مارک کردن intent به عنوان restake_applied
    meta["restake_applied"] = True
    meta["restake_applied_at"] = timezone.now().isoformat()
    intent.meta = meta
    # اگر می‌خواهی صراحتاً confirmed باشد:
    intent.status = "confirmed"
    intent.save(update_fields=["meta", "status", "updated_at"])
    stake.created_at = timezone.now()
    stake.save(update_fields=["created_at"])

    # 6) تسک و بوسترها
    try:
        complete_weekly_restake_task(user)
    except Exception:
        pass

    try:
        evaluate_and_reward_for_user(user)
    except Exception:
        pass


def _default_required_days_for_task(task: Task) -> int:
    if task.kind == TaskKind.WEEKLY:
        return 7
    if task.kind == TaskKind.MONTHLY:
        # می‌تونی 30، یا 31، یا دینامیک از روی window_start/window_end بگیری
        return 30
    # برای EVENT / ONE_TIME اگر خواستی می‌تونی چیز دیگری بزاری
    return 7


def get_current_window(task: Task):
    return _current_window(task)


def _current_window(task: Task) -> tuple[timezone.datetime | None, timezone.datetime | None]:
    now = timezone.now()

    if task.kind == TaskKind.ONE_TIME:
        return (None, None)

    if task.kind == TaskKind.WEEKLY:
        # امروز در تایم‌زون لوکال
        today_local = timezone.localdate()
        # دوشنبه هفته
        week_start_date = today_local - timedelta(days=today_local.weekday())
        # شروع هفته: دوشنبه ساعت 00:00
        start = timezone.make_aware(datetime.combine(week_start_date, time.min))
        end = start + timedelta(days=7)
        return (start, end)

    if task.kind == TaskKind.MONTHLY:
        start = now.replace(day=1, hour=0, minute=0, second=0, microsecond=0)
        if start.month == 12:
            end = start.replace(year=start.year + 1, month=1)
        else:
            end = start.replace(month=start.month + 1)
        return (start, end)

    return (task.window_start, task.window_end)


def _cond_login_streak(user: User, params: dict, window: tuple) -> bool:
    """
    True فقط وقتی که واقعا streak به حد نصاب برسد
    """
    # یک‌دست کردن N
    days_required = int(
        params.get("required_days")
        or params.get("days")
        or params.get("required_count")
        or _default_required_days_for_task(task)
    )

    end = timezone.localdate()
    streak = 0
    current = end

    while streak < days_required:
        if UserDailyLogin.objects.filter(user=user, date=current).exists():
            streak += 1
            current = current - timedelta(days=1)
        else:
            break

    return streak >= days_required


def _cond_weekly_restake_all(user: User, params: dict, window: tuple) -> bool:
    """در بازه weekly یک عملیات restake داشته باشد."""
    start, end = window
    # اگر تراکنش restake ثبت دارید، اینجا چک کنید؛ فعلاً ساده:
    return bool(params.get("demo_pass", True))


def _cond_miners_active_window(user: User, params: dict, window: tuple) -> bool:
    """
       Condition is satisfied if the UserTaskProgress for this window is COMPLETED.
       را معمولاً evaluate_and_reward_for_user بر اساس همین استفاده می‌کند.
       """
    from apps.events.models import UserTaskProgress, ProgressStatus

    prg = (
        UserTaskProgress.objects.filter(
            user=user,
            task__condition_code=TaskConditionCode.MINERS_ACTIVE_WINDOW,
            window_start=window[0],
            window_end=window[1],
        )
            .order_by("-window_start", "-id")
            .first()
    )


    return bool(prg and prg.status == ProgressStatus.COMPLETED)


def _cond_referral_active_count(user: User, params: dict, window: tuple) -> bool:
    n_required = int(params.get("n", 5))
    from apps.users.services.referrals import count_active_referrals

    return count_active_referrals(user) >= n_required


_COND_MAP = {
    TaskConditionCode.LOGIN_STREAK: _cond_login_streak,
    TaskConditionCode.WEEKLY_RESTAKE_ALL: _cond_weekly_restake_all,
    TaskConditionCode.MINERS_ACTIVE_WINDOW: _cond_miners_active_window,
    TaskConditionCode.REFERRAL_ACTIVE_COUNT: _cond_referral_active_count,
}


# --- Reward activations ---



def _activate_energy_turbo(user, task, now=None):
    """
    Create UserEnergyBoost based on task.reward_object (EnergyTurboDefinition).
    """
    now = now or timezone.now()

    if task.reward_type != TaskRewardType.ENERGY_TURBO:
        return None

    definition = getattr(task, "reward_object", None)
    if not isinstance(definition, EnergyTurboDefinition):
        return None

    ends_at = now + timezone.timedelta(seconds=definition.duration_seconds)

    boost = UserEnergyBoost.objects.create(
        user=user,
        definition=definition,
        starts_at=now,
        ends_at=ends_at,
    )
    return boost



def _activate_miner_access(user: User, defn: MinerAccessDefinition) -> UserTemporaryMinerAccess:
    now = timezone.now()
    ends = now + timedelta(seconds=int(defn.duration_seconds))
    return UserTemporaryMinerAccess.objects.create(
        user=user, definition=defn, starts_at=now, ends_at=ends, is_active=True
    )




def expire_time_limited_rewards() -> int:
    """بوست‌ها و دسترسی‌های منقضی‌شده را غیرفعال می‌کند. خروجی: تعداد رکوردهای deactivated."""
    now = timezone.now()
    n = 0
    n += UserEnergyBoost.objects.filter(is_active=True, ends_at__lt=now).update(is_active=False)
    n += UserTemporaryMinerAccess.objects.filter(is_active=True, ends_at__lt=now).update(
        is_active=False
    )
    return n


def complete_weekly_restake_task(user):
    """وقتی کاربر ری‌استیک کرد، تسک مربوطه را complete کن (اگر وجود دارد)."""
    try:
        task = Task.objects.filter(
            condition_code=TaskConditionCode.WEEKLY_RESTAKE_ALL, is_active=True
        ).first()
        if not task:
            return False

        window = _current_window(task)
        prg, _ = UserTaskProgress.objects.get_or_create(
            user=user,
            task=task,
            window_start=window[0],
            window_end=window[1],
            defaults={"status": ProgressStatus.PENDING},
        )
        prg.status = ProgressStatus.COMPLETED
        prg.completed_at = timezone.now()
        prg.save(update_fields=["status", "completed_at"])
        return True
    except Exception:
        return False


def register_login_activity(user: User) -> None:
    """
    هر بار که یوزر لاگین می‌کند:
    1) یک UserDailyLogin برای امروز ثبت می‌کنیم.
    2) تسک‌هایی که condition_code = LOGIN_STREAK دارند را آپدیت می‌کنیم:
       - current_streak (چند روز پشت سر هم)
       - progress_percent = streak / days * 100
    """
    today = timezone.localdate()
    UserDailyLogin.objects.get_or_create(user=user, date=today)

    tasks = Task.objects.filter(
        is_active=True,
        condition_code=TaskConditionCode.LOGIN_STREAK,
    )

    for task in tasks:
        params = task.condition_params or {}

        days_required = int(
            params.get("required_days")
            or params.get("days")
            or params.get("required_count")
            or _default_required_days_for_task(task)
        )

        # محاسبه streak رو به عقب
        streak = 0
        current = today
        while streak < days_required:
            if UserDailyLogin.objects.filter(user=user, date=current).exists():
                streak += 1
                current = current - timedelta(days=1)
            else:
                break

        window = _current_window(task)
        prg, _ = UserTaskProgress.objects.get_or_create(
            user=user,
            task=task,
            window_start=window[0],
            window_end=window[1],
            defaults={"status": ProgressStatus.PENDING},
        )

        progress_percent = min(streak, days_required) / days_required * 100

        meta = prg.meta or {}
        meta.update(
            {
                "current_streak": streak,
                "required_days": days_required,
                "progress_percent": progress_percent,
            }
        )
        prg.meta = meta

        if streak >= days_required and prg.status == ProgressStatus.PENDING:
            prg.status = ProgressStatus.COMPLETED
            prg.completed_at = timezone.now()

        prg.save()


def update_login_streak_progress(user):
    """
    Update login streak progress for all login_streak tasks for the given user.
    This function should be called once per day when the user is active
    (for example, when /api/events/dashboard/ is requested).
    """
    today = timezone.now().date()
    yesterday = today - timedelta(days=1)

    # All active tasks that are login_streak type
    tasks = Task.objects.filter(
        is_active=True,
        condition_code=TaskConditionCode.LOGIN_STREAK,
    )

    for task in tasks:
        params = task.condition_params or {}
        required_days = int(
            params.get("required_days")
            or params.get("days")
            or params.get("required_count")
            or _default_required_days_for_task(task)
        )

        # Get current window for this task (weekly/monthly etc.)
        window_start, window_end = get_current_window(task)

        progress, _ = UserTaskProgress.objects.get_or_create(
            user=user,
            task=task,
            window_start=window_start,
            window_end=window_end,
            defaults={"status": ProgressStatus.PENDING},
        )

        meta = progress.meta or {}
        last_login_str = meta.get("last_login_date")
        current_streak = int(meta.get("current_streak", 0))

        if last_login_str == str(today):
            # Already counted today -> nothing to do
            continue
        elif last_login_str == str(yesterday):
            # Consecutive day -> increase streak
            current_streak += 1
        else:
            # Broken streak or first time -> reset to 1
            current_streak = 1

        progress_percent = min(current_streak, required_days) / required_days * 100

        meta.update(
            {
                "last_login_date": str(today),
                "current_streak": current_streak,
                "required_days": required_days,
                "progress_percent": progress_percent,
            }
        )
        progress.meta = meta

        if current_streak >= required_days and progress.status == ProgressStatus.PENDING:
            progress.status = ProgressStatus.COMPLETED
            progress.completed_at = timezone.now()

        progress.save()


def ensure_login_activity_for_today(user: User):
    today = timezone.localdate()
    if not UserDailyLogin.objects.filter(user=user, date=today).exists():
        register_login_activity(user)
        evaluate_and_reward_for_user(user)






def _get_required_count_for_task(task: Task) -> int:
    """
    Read N from condition_params with backward compatibility.
    Fallback to kind-based defaults if not provided.

    از کلیدهای مختلفی مثل n / days / required_days هم پشتیبانی می‌کند،
    و نسبت به حروف بزرگ/کوچک حساس نیست.
    """
    raw_params = task.condition_params or {}

    # نرمال‌سازی keyها به lowercase
    params = {}
    for k, v in raw_params.items():
        if isinstance(k, str):
            params[k.lower().strip()] = v
        else:
            params[k] = v

    rc = (
        params.get("required_count")
        or params.get("required_days")
        or params.get("days")
        or params.get("n")
    )

    if rc is not None:
        try:
            # اگر rc رشته با اسپیس باشد هم درست parse شود
            return int(str(rc).strip())
        except (TypeError, ValueError):
            pass

    # Fallback based on kind
    if task.kind == TaskKind.WEEKLY:
        return 7
    if task.kind == TaskKind.MONTHLY:
        return 30

    # Default for other kinds
    return 1


def _are_all_miners_active(user) -> bool:
    """
    Returns True if all of the user's miners are online.
    """
    qs = UserMiner.objects.filter(user=user)
    if not qs.exists():
        return False  # یا اگر تصمیم‌تان چیز دیگری است، اینجا عوضش کن

    inactive_exists = qs.filter(is_online=False).exists()
    return not inactive_exists

def update_miners_active_window_progress(user) -> None:
    """
    Update progress for all MINERS_ACTIVE_WINDOW tasks for the given user
    for today's date.

    - اگر امروز قبلاً برای آن تسک آپدیت شده، دوباره تغییر نمی‌دهیم (idempotent per-day).
    - اگر امروز همه ماینرها active باشند، days_ok++.
    - progress_percent = days_ok / required_days * 100.
    - اگر days_ok >= required_days → status = COMPLETED.
    """

    today = timezone.localdate()

    tasks = Task.objects.filter(
        is_active=True,
        condition_code=TaskConditionCode.MINERS_ACTIVE_WINDOW,
    )

    if not tasks.exists():
        return

    all_active_today = _are_all_miners_active(user)

    for task in tasks:
        window = _current_window(task)  # فرض می‌کنم الان در کدت داری این helper رو
        if not _date_in_window(today, window):
            continue

        required_days = _get_required_count_for_task(task)

        prg, _ = UserTaskProgress.objects.get_or_create(
            user=user,
            task=task,
            window_start=window[0],
            window_end=window[1],
            defaults={"status": ProgressStatus.PENDING},
        )

        meta = prg.meta or {}
        last_update_date = meta.get("last_update_date")
        old_required_days = int(meta.get("required_days", required_days))

        # اگر امروز را قبلاً حساب کردیم، دیگه دست نمی‌زنیم
        if last_update_date == str(today):
            if old_required_days != required_days:
                days_ok = int(meta.get("days_ok", 0))
                progress_percent = min(days_ok, required_days) / required_days * 100

                meta.update(
                    {
                        "required_days": required_days,
                        "progress_percent": progress_percent,
                    }
                )
                prg.meta = meta
                prg.save(update_fields=["meta"])
            continue

        days_ok = int(meta.get("days_ok", 0))

        if all_active_today:
            days_ok += 1

        progress_percent = min(days_ok, required_days) / required_days * 100

        meta.update(
            {
                "days_ok": days_ok,
                "required_days": required_days,
                "last_update_date": str(today),
                "progress_percent": progress_percent,
            }
        )

        prg.meta = meta

        if days_ok >= required_days and prg.status == ProgressStatus.PENDING:
            prg.status = ProgressStatus.COMPLETED
            prg.completed_at = timezone.now()

        prg.save()