from rest_framework import serializers

from django.utils import timezone

from apps.events.models import Task, UserEnergyBoost, UserTaskProgress, UserTemporaryMinerAccess
from apps.events.serializers import EnergyBoostSerializer, MinerAccessSerializer
from apps.events.models import (
    Task, UserTaskProgress, UserEnergyBoost, UserTemporaryMinerAccess, ProgressStatus
)
from apps.events.services import get_current_window


class UserTaskProgressSerializer(serializers.ModelSerializer):
    task_title = serializers.CharField(source="task.title", read_only=True)
    reward_type = serializers.CharField(source="task.reward_type", read_only=True)
    progress_percent = serializers.SerializerMethodField()
    class Meta:
        model = UserTaskProgress
        fields = [
            "id",
            "task",
            "task_title",
            "reward_type",
            "status",
            "completed_at",
            "rewarded_at",
            "window_start",
            "window_end",
            "progress_percent",
        ]

    def get_progress_percent(self, obj: UserTaskProgress):
        meta = obj.meta or {}
        val = meta.get("progress_percent")
        if val is None:
            # سازگاری با قدیمی‌ها: اگر COMPELTED باشد، 100 را برگردان
            return 100 if obj.status in ("completed", "rewarded") else 0
        try:
            return round(float(val), 2)
        except Exception:
            return 0


class TaskWithProgressSerializer(serializers.ModelSerializer):
    image = serializers.ImageField(read_only=True)
    status = serializers.SerializerMethodField()
    progress_percent = serializers.SerializerMethodField()

    class Meta:
        model = Task
        fields = [
            "id",
            "slug",
            "title",
            "description",
            "kind",
            "reward_type",
            "image",
            "status",
            "progress_percent",
        ]

    def _get_progress(self, obj):
        request = self.context.get("request")
        user = getattr(request, "user", None)
        if not user or not user.is_authenticated:
            return None

        # فقط پروگرس همین بازه‌ی فعلی رو بگیر
        window_start, window_end = get_current_window(obj)

        qs = UserTaskProgress.objects.filter(user=user, task=obj)
        if window_start and window_end:
            qs = qs.filter(window_start=window_start, window_end=window_end)

        return qs.order_by("-id").first()

    def get_status(self, obj):
        progress = self._get_progress(obj)
        if not progress:
            return ProgressStatus.PENDING
        return progress.status

    def get_progress_percent(self, obj):
        progress = self._get_progress(obj)
        if not progress:
            return 0

        meta = progress.meta or {}
        if "progress_percent" in meta:
            return meta["progress_percent"]

        if progress.status in (ProgressStatus.COMPLETED, ProgressStatus.REWARDED):
            return 100

        return 0

    # def get_progress_percent(self, obj):
    #     prg = self._get_progress(obj)
    #     if not prg:
    #         return 0
    #
    #     meta = prg.meta or {}
    #
    #     # اگر درصد در meta هست → همان را برگردان
    #     if "progress_percent" in meta:
    #         try:
    #             return float(meta["progress_percent"])
    #         except:
    #             pass
    #
    #     # اگر تسک one_time است و completed شده → 100%
    #     if obj.kind == "one_time" and prg.status in (ProgressStatus.COMPLETED, ProgressStatus.REWARDED):
    #         return 100
    #
    #     # در غیر این صورت (weekly, monthly, event) درصد باید از meta بیاید
    #     return 0


class EventsDashboardSerializer(serializers.Serializer):
    """نمایش داده‌های ترکیبی برای داشبورد یوزر"""

    tasks = TaskWithProgressSerializer(many=True)
    active_boosts = EnergyBoostSerializer(many=True)
    active_miner_accesses = MinerAccessSerializer(many=True)
    # task_progress = UserTaskProgressSerializer(many=True)  # برای سازگاری، اگر دوست نداشتی می‌تونیم حذفش کنیم
    now = serializers.DateTimeField(default=timezone.now)
    current_percent = serializers.DecimalField(max_digits=8, decimal_places=4)
