from django.conf import settings
from django.contrib.contenttypes.fields import GenericForeignKey
from django.contrib.contenttypes.models import ContentType
from django.core.validators import MinValueValidator
from django.db import models
from django.utils import timezone

from .constants import ProgressStatus, RewardType, TaskConditionCode, TaskKind

User = settings.AUTH_USER_MODEL


# ---------- Reward Definitions (admin-configurable) ----------


class EnergyTurboDefinition(models.Model):
    name = models.CharField(max_length=120)
    # درصد افزایش سود (مثلاً 10 یعنی +10%)
    percent_boost = models.DecimalField(
        max_digits=6, decimal_places=2, validators=[MinValueValidator(0)]
    )
    # مدت اثر (ثانیه)
    duration_seconds = models.PositiveIntegerField(default=3600)
    image = models.ImageField(upload_to="rewards/energy/", null=True, blank=True)
    is_active = models.BooleanField(default=True)

    class Meta:
        verbose_name = "Energy Turbo"
        verbose_name_plural = "Energy Turbos"

    def __str__(self):
        return f"{self.name} (+{self.percent_boost}% / {self.duration_seconds}s)"


class MinerAccessDefinition(models.Model):
    """
    ماینر سطح‌بالا که ادمین از بین Miners انتخاب می‌کند (apps.miners.Miner).
    """

    name = models.CharField(max_length=120)
    miner = models.ForeignKey("miners.Miner", on_delete=models.PROTECT, related_name="reward_defs")
    duration_seconds = models.PositiveIntegerField(default=3600)
    image = models.ImageField(upload_to="rewards/miner_access/", null=True, blank=True)
    is_active = models.BooleanField(default=True)

    class Meta:
        verbose_name = "Premium Miner Access"
        verbose_name_plural = "Premium Miner Accesses"

    def __str__(self):
        return f"{self.name} -> {self.miner} ({self.duration_seconds}s)"


# ---------- Tasks (admin-configurable) ----------


class Task(models.Model):
    slug = models.SlugField(unique=True, max_length=120)
    title = models.CharField(max_length=180)
    description = models.TextField(blank=True)
    kind = models.CharField(max_length=20, choices=TaskKind.choices, default=TaskKind.ONE_TIME)

    # بازه زمانی سفارشی برای نوع EVENT (یا برای محدودکردن weekly/monthly)
    window_start = models.DateTimeField(null=True, blank=True)
    window_end = models.DateTimeField(null=True, blank=True)
    image = models.ImageField(
        upload_to="events/tasks/",
        null=True,
        blank=True,
        help_text="image for this task",
    )
    # تعریف شرط به‌صورت کُد و پارامترها
    condition_code = models.CharField(max_length=40, choices=TaskConditionCode.choices)
    condition_params = models.JSONField(default=dict, blank=True)
    is_active = models.BooleanField(default=True)

    # انتخاب نوع ریوارد + مرجع تعریف ریوارد
    reward_type = models.CharField(max_length=20, choices=RewardType.choices)
    # Generic to either EnergyTurboDefinition or MinerAccessDefinition
    reward_content_type = models.ForeignKey(
        ContentType, on_delete=models.PROTECT, null=True, blank=True
    )
    reward_object_id = models.PositiveIntegerField(null=True, blank=True)
    reward_object = GenericForeignKey("reward_content_type", "reward_object_id")

    created_at = models.DateTimeField(default=timezone.now)

    class Meta:
        indexes = [models.Index(fields=["kind", "is_active"])]
        verbose_name = "Task"
        verbose_name_plural = "Tasks"

    def __str__(self):
        return self.title


# ---------- User Progress & Active Rewards ----------


class UserTaskProgress(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name="task_progress")
    task = models.ForeignKey(Task, on_delete=models.CASCADE, related_name="user_progress")
    status = models.CharField(
        max_length=20, choices=ProgressStatus.choices, default=ProgressStatus.PENDING
    )
    window_start = models.DateTimeField(null=True, blank=True)
    window_end = models.DateTimeField(null=True, blank=True)
    completed_at = models.DateTimeField(null=True, blank=True)
    rewarded_at = models.DateTimeField(null=True, blank=True)
    meta = models.JSONField(default=dict, blank=True)  # جزئیات محاسبه/ایونت‌ها

    class Meta:
        unique_together = [("user", "task", "window_start", "window_end")]
        indexes = [models.Index(fields=["status", "window_end"])]

    def __str__(self):
        return f"{self.user} - {self.task} [{self.status}]"


class UserEnergyBoost(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name="energy_boosts")
    definition = models.ForeignKey(EnergyTurboDefinition, on_delete=models.PROTECT)
    # اگر None باشد یعنی روی کل ماینرهای کاربر اعمال می‌شود؛ در آینده می‌توان miner خاص را ست کرد.
    miner = models.ForeignKey(
        "miners.UserMiner",
        on_delete=models.CASCADE,
        null=True,
        blank=True,
        related_name="energy_boosts",
    )
    starts_at = models.DateTimeField()
    ends_at = models.DateTimeField()
    is_active = models.BooleanField(default=True)

    class Meta:
        indexes = [models.Index(fields=["user", "is_active", "ends_at"])]


class UserTemporaryMinerAccess(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name="temp_miner_accesses")
    definition = models.ForeignKey(MinerAccessDefinition, on_delete=models.PROTECT)
    starts_at = models.DateTimeField()
    ends_at = models.DateTimeField()
    is_active = models.BooleanField(default=True)

    class Meta:
        indexes = [models.Index(fields=["user", "is_active", "ends_at"])]

class UserDailyLogin(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name="daily_logins")
    date = models.DateField()
    created_at = models.DateTimeField(auto_now_add=True)

    class Meta:
        unique_together = [("user", "date")]
        indexes = [models.Index(fields=["user", "date"])]

    def __str__(self):
        return f"{self.user_id} @ {self.date}"
