# apps/config/management/commands/set_referral_bonus.py
from django.core.management.base import BaseCommand, CommandError

try:
    from apps.config.models import Config
except Exception:
    Config = None

PER_ACTIVE_KEY = "REFERRAL_PER_ACTIVE_BONUS_BP"  # e.g. 1 => 0.01%
CAP_KEY = "REFERRAL_BONUS_CAP_BP"  # e.g. 150 => 1.50%


class Command(BaseCommand):
    help = "Set or show referral bonus settings (basis points)."

    def add_arguments(self, parser):
        parser.add_argument(
            "--per-active", type=int, help="Per active referral (bp), e.g. 1 = 0.01%"
        )
        parser.add_argument("--cap", type=int, help="Cap (bp), e.g. 150 = 1.50%")
        parser.add_argument("--show", action="store_true", help="Only show current values")

    def handle(self, *args, **opts):
        if Config is None:
            raise CommandError("apps.config.models.Config not found. Check your project structure.")

        def _get(key, default=None):
            obj = Config.objects.filter(key=key).first()
            return obj.value if obj else default

        if opts["show"]:
            self.stdout.write(
                self.style.SUCCESS(
                    f"{PER_ACTIVE_KEY}={_get(PER_ACTIVE_KEY, 'N/A')} ; {CAP_KEY}={_get(CAP_KEY, 'N/A')}"
                )
            )
            return

        updated = []
        if opts["per_active"] is not None:
            obj, _ = Config.objects.update_or_create(
                key=PER_ACTIVE_KEY, defaults={"value": str(opts["per_active"])}
            )
            updated.append(f"{PER_ACTIVE_KEY}={obj.value}")
        if opts["cap"] is not None:
            obj, _ = Config.objects.update_or_create(
                key=CAP_KEY, defaults={"value": str(opts["cap"])}
            )
            updated.append(f"{CAP_KEY}={obj.value}")

        if updated:
            self.stdout.write(self.style.SUCCESS("Updated: " + ", ".join(updated)))
        else:
            self.stdout.write(
                self.style.WARNING("Nothing updated. Use --per-active/--cap or --show.")
            )
