# apps/claims/utils.py
from decimal import Decimal

def human_to_scaled_uint(amount_human, decimals):
    """
    amount_human: Decimal
    return: int (uint256)
    """
    q = Decimal(10) ** Decimal(decimals)
    # floor (بدون رندینگ به بالا)
    scaled = (Decimal(amount_human) * q).to_integral_value(rounding="ROUND_DOWN")
    if scaled < 0:
        scaled = Decimal(0)
    return int(scaled)

from typing import Optional, Dict, Any
from eth_utils import to_checksum_address

# امضای ایونت Claimed:
CLAIMED_TOPIC0 = "0x2f6639d24651730c7bf57c95ddbf96d66d11477e4ec626876f92c22e5f365e68"

def _slice32(data_hex: str, idx: int) -> str:
    # data_hex: "0x...." -> بدون 0x و برش 32بایتی
    h = data_hex[2:] if data_hex.startswith("0x") else data_hex
    return h[idx*64:(idx+1)*64]

def _as_address(word32: str) -> str:
    # آخرین 40 کاراکتر (20 بایت) آدرس است
    return to_checksum_address("0x" + word32[-40:])

def _as_uint(word32: str) -> int:
    return int(word32, 16)

def parse_claimed_log(log: Dict[str, Any]) -> Optional[Dict[str, Any]]:
    """
    اگر log مربوط به Claimed باشد، دیکد می‌کند و dict برمی‌گرداند.
    خروجی:
    {
      'tx_hash', 'user', 'token', 'amount_scaled', 'nonce', 'log_address'
    }
    """
    topics = log.get("topics") or []
    if not topics or topics[0].lower() != CLAIMED_TOPIC0:
        return None

    data = log.get("data", "")
    if not data or len(data) < 2 + 64*4:
        return None

    w0 = _slice32(data, 0)  # user
    w1 = _slice32(data, 1)  # token
    w2 = _slice32(data, 2)  # amount
    w3 = _slice32(data, 3)  # nonce

    user  = _as_address(w0)
    token = _as_address(w1)
    amount_scaled = _as_uint(w2)
    nonce = _as_uint(w3)

    return {
        "tx_hash": log.get("transactionHash"),
        "log_address": log.get("address"),
        "user": user,
        "token": to_checksum_address(token),
        "amount_scaled": amount_scaled,
        "nonce": nonce,
    }