from django.core.management.base import BaseCommand
from django.conf import settings
from django.utils import timezone
from web3 import Web3
from apps.claims.models import ClaimTx

class Command(BaseCommand):
    help = "Confirm pending ClaimTx by checking receipts via QuickNode HTTP"

    def handle(self, *args, **opts):
        http_url = getattr(settings, "QUICKNODE_HTTP_URL", "")
        if not http_url:
            self.stderr.write("QUICKNODE_HTTP_URL not set.")
            return

        w3 = Web3(Web3.HTTPProvider(http_url))
        required_conf = int(getattr(settings, "WITHDRAW_REQUIRED_CONF", 2))

        cnt_ok, cnt_fail, cnt_skip = 0, 0, 0

        # تراکنش‌های در انتظار
        qs = ClaimTx.objects.filter(status="pending").order_by("id")[:200]

        latest = w3.eth.block_number
        for obj in qs:
            try:
                if not obj.tx_hash:
                    cnt_skip += 1
                    continue
                rc = w3.eth.get_transaction_receipt(obj.tx_hash)
                if not rc:
                    cnt_skip += 1
                    continue

                # وضعیت receipt
                status = rc.get("status", 0)  # 1 success, 0 fail
                blk = rc.get("blockNumber")
                confs = (latest - blk) if (blk and latest) else 0

                if status == 1 and confs >= required_conf:
                    obj.status = "confirmed"
                    obj.updated_at = timezone.now()
                    # اگر amount_scaled و token_address از webhook پر نشده بود، از لاگ event (topics/data) بخوان—در صورت نیاز
                    obj.save(update_fields=["status", "updated_at"])
                    cnt_ok += 1
                elif status == 0:
                    obj.status = "failed"
                    obj.updated_at = timezone.now()
                    obj.save(update_fields=["status", "updated_at"])
                    cnt_fail += 1
                else:
                    cnt_skip += 1
            except Exception:
                cnt_skip += 1

        self.stdout.write(self.style.SUCCESS(
            f"confirmed={cnt_ok} failed={cnt_fail} skipped={cnt_skip}, latest_block={latest}"
        ))