import csv

from django.contrib import admin
from django.http import HttpResponse
from django.template.response import TemplateResponse
from django.urls import path
from django.utils import timezone
from django.utils.html import format_html

from .models import CampaignStatus, EmailCampaign
from .services import schedule_campaign


@admin.register(EmailCampaign)
class EmailCampaignAdmin(admin.ModelAdmin):
    list_display = ["id", "subject", "status", "created_at", "scheduled_at", "sent_count", "total_recipients"]
    readonly_fields = ["status", "sent_count", "total_recipients", "error_count", "created_at"]
    fields = [
        "subject",
        "text_content",
        "image",
        "attachment",
        "status",
        "scheduled_at",
        "created_at",
        "sent_count",
        "total_recipients",
        "error_count",
    ]
    actions = ["start_sending", "export_errors"]
    change_form_template = "admin/broadcast/campaign_change_form.html"
    def start_sending(self, request, queryset):
        for campaign in queryset:
            if campaign.status != CampaignStatus.DRAFT:
                continue
            schedule_campaign(campaign)
        self.message_user(request, "Selected campaigns scheduled for sending.")

    start_sending.short_description = "Start sending selected campaigns"

    def get_urls(self):
        urls = super().get_urls()
        custom = [
            path(
                "<int:campaign_id>/preview/",
                self.admin_site.admin_view(self.preview_view),
                name="broadcast_preview",
            ),
        ]
        return custom + urls

    def preview_view(self, request, campaign_id, *args, **kwargs):
        from django.contrib.auth import get_user_model
        from django.shortcuts import get_object_or_404
        from django.template.loader import render_to_string

        from .models import EmailCampaign

        User = get_user_model()
        campaign = get_object_or_404(EmailCampaign, pk=campaign_id)
        fake_user = User.objects.filter(is_active=True).first()

        html = render_to_string(
            "email/emails/templates/base_campaign.html",
            {
                "campaign": campaign,
                "text_content": campaign.text_content,
                "subject": campaign.subject,
                "image_url": request.build_absolute_uri(campaign.image.url) if campaign.image else None,                "username": getattr(fake_user, "username", "demo_user"),
                "site_name": "CoinMaining Game",
            },
        )
        return TemplateResponse(
            request, "admin/broadcast/preview.html", {"html": html, "campaign": campaign}
        )

    @admin.action(description="Export error deliveries to CSV")
    def export_errors(self, request, queryset):
        from .models import EmailDelivery

        rows = EmailDelivery.objects.filter(
            campaign__in=queryset, error_message__isnull=False
        ).exclude(error_message="")

        response = HttpResponse(content_type="text/csv")
        response["Content-Disposition"] = 'attachment; filename="email_errors.csv"'
        writer = csv.writer(response)
        writer.writerow(["Campaign", "User", "Email", "Error"])
        for r in rows:
            writer.writerow([r.campaign.subject, r.user.username, r.user.email, r.error_message])
        return response
