# apps/badges/models.py
from django.db import models
from django.conf import settings

class Badge(models.Model):
    class Rarity(models.TextChoices):
        COMMON = "COMMON", "Common"
        UNCOMMON = "UNCOMMON", "Uncommon"
        RARE = "RARE", "Rare"
        EPIC = "EPIC", "Epic"
        
    TYPE_CHOICES = [
        ("GENERIC", "Generic/Manual"),
        ("REFERRAL", "Referral Threshold"),
        ("WHALE", "Whale (no-unstake period)"),
        ("LEVEL5", "Started a Level-5 Miner"),
        ("TOP_ROI", "Top ROI Achiever"),
        ("FIRST_N", "First N Users (giveaway)"),
    ]

    key = models.SlugField(max_length=64, unique=True)   # مثل: referral_5, referral_10, whale_6m, level5_starter, top_roi_15, first_100
    name = models.CharField(max_length=128)
    description = models.TextField(blank=True)
    image = models.ImageField(upload_to="badges/", blank=True, null=True)

    # بونس به bp (basis points): 10 => 0.10%
    reward_rate_bp = models.PositiveIntegerField(default=0)

    type = models.CharField(max_length=16, choices=TYPE_CHOICES, default="GENERIC")
    is_active = models.BooleanField(default=True)

    # پارامترهای نوع بج (برای REFERRAL: threshold، برای FIRST_N: limit_count، برای TOP_ROI: roi_threshold_bp، برای WHALE: months)
    limit_count = models.PositiveIntegerField(null=True, blank=True)      # برای FIRST_N
    referral_threshold = models.PositiveIntegerField(null=True, blank=True)  # 5,10,25,50
    whale_months = models.PositiveIntegerField(null=True, blank=True)     # مثلا 6
    roi_threshold_bp = models.PositiveIntegerField(null=True, blank=True) # 150 => ROI 1.5x
    sort_order = models.PositiveIntegerField(default=0, db_index=True)
    # متفرقه
    created_at = models.DateTimeField(auto_now_add=True)
    rarity = models.CharField(
        max_length=16,
        choices=Rarity.choices,
        default=Rarity.COMMON,
        db_index=True,
    )
    display_order = models.PositiveIntegerField(default=0, db_index=True)

    
    ordering = ["rarity", "display_order", "-created_at"]
    class Meta:
        indexes = [
            models.Index(fields=["type"]),
            models.Index(fields=["is_active"]),
            models.Index(fields=["rarity", "display_order"]),
        ]

    def __str__(self):
        return f"{self.key} ({self.name})"


class UserBadge(models.Model):
    user = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.CASCADE)
    badge = models.ForeignKey(Badge, on_delete=models.CASCADE)
    awarded_at = models.DateTimeField(auto_now_add=True)
    is_active = models.BooleanField(default=True)  # برای revoke احتمالی
    # جلوگیری از دوبار اعطا
    class Meta:
        unique_together = [("user", "badge")]
        indexes = [
            models.Index(fields=["user"]),
            models.Index(fields=["badge"]),
        ]

    def __str__(self):
        return f"{self.user_id}:{self.badge.key}"