# apps/wallets/service.py
# --- Standard Library
from decimal import Decimal, InvalidOperation

# --- Django
from django.db import transaction
from django.db.models import F, Sum
from django.utils import timezone

# --- Project
from apps.stakes.models import Stake
from apps.stakes.services import accrue_until_now
from apps.wallets.models import PendingReward

# Optional (اگر مدل WithdrawRequest در پروژه نباشه، ارور نده)
try:
    from apps.wallets.models import WithdrawRequest
except Exception:
    WithdrawRequest = None

from collections import defaultdict

from django.conf import settings

from apps.config.models import Config  # مسیر واقعی مدل Config را جایگزین کن

# سیاست‌ها (بعداً اگر لازم شد از settings بخونیم)
MIN_WITHDRAW = Decimal("0.0001")
MAX_WITHDRAW = Decimal("1000000000")  # سقف محافظه‌کارانه


def get_withdrawable_amount(user) -> Decimal:
    """
    مقدار قابل‌برداشت: جمع PendingRewardهایی که claimed هستند و هنوز withdrawn=False دارند.
    """
    if not user or not getattr(user, "id", None):
        return Decimal("0")
    total = (
        PendingReward.objects.filter(user=user, status="claimed", withdrawn=False).aggregate(
            total=Sum("amount")
        )["total"]
    ) or Decimal("0")
    return total.quantize(Decimal("0.000000000000000001"))


def user_has_pending_withdraw(user) -> bool:
    """
    اگر مدل WithdrawRequest و وضعیت‌های رایج موجود باشند، بررسی می‌کنیم؛ در غیر اینصورت False.
    """
    if not WithdrawRequest:
        return False
    return WithdrawRequest.objects.filter(user=user, status__in=("pending", "processing")).exists()


def validate_withdraw_request(user, amount: Decimal) -> None:
    """
    فقط اعتبارسنجی بیزنسی انجام می‌دهد. (هیچ تغییری در DB نمی‌دهد)
    خطاها را با ValueError بالا می‌اندازد تا ویو آن را هندل کند.
    """
    if amount <= 0:
        raise ValueError("Amount must be positive.")
    if amount < MIN_WITHDRAW:
        raise ValueError(f"Amount must be >= {MIN_WITHDRAW}.")
    if amount > MAX_WITHDRAW:
        raise ValueError(f"Amount must be <= {MAX_WITHDRAW}.")
    if user_has_pending_withdraw(user):
        raise ValueError("You have a pending withdraw request.")
    withdrawable = get_withdrawable_amount(user)
    if amount > withdrawable:
        raise ValueError("Insufficient withdrawable balance.")
    # همه‌چی اوکی


@transaction.atomic
def create_withdraw_and_reserve(user, amount: Decimal):
    """
    1) اعتبارسنجی بیزنسی
    2) قفل/رزرو منابع لازم
    3) ایجاد درخواست برداشت
    """
    validate_withdraw_request(user, amount)

    # ✅ انتخاب پاداش‌های قابل برداشت و رزرو آن‌ها
    # (اگر مدل/فیلد دیگری برای رزرو داری، اینجا مطابقش کن)
    rewards_qs = (
        PendingReward.objects.select_for_update(skip_locked=True)
        .filter(user=user, status="claimed", withdrawn=False)
        .order_by("id")
    )

    remaining = amount
    picked_ids = []
    total_picked = Decimal("0")

    for r in rewards_qs:
        if remaining <= 0:
            break
        take = r.amount if r.amount <= remaining else remaining
        # اینجا می‌تونی فیلد «locked_amount» داشته باشی؛ اگر نداری، در انتها فلگ withdrawn را ست می‌کنیم.
        picked_ids.append(r.id)
        total_picked += take
        remaining -= take

    if total_picked < amount:
        # مسابقهٔ هم‌زمان یا تغییر داده‌ها
        raise ValueError("Insufficient withdrawable balance (race detected).")

    # ✅ ایجاد رکورد WithdrawRequest (اگر مدل داری)
    req_id = None
    if WithdrawRequest:
        req = WithdrawRequest.objects.create(
            user=user,
            amount=amount,
            status="pending",  # یا status اولیه‌ی پروژه‌ات
            meta={"picked_rewards": picked_ids},
        )
        req_id = req.id

    # ✅ علامت‌گذاری rewards رزرو شده (ساده‌ترین: withdrawn=True)
    # اگر «بخشی» از یک reward را باید رزرو کنی، بهتر است فیلد locked_amount داشته باشی؛
    # اینجا برای سادگی، همه را برداشتنی فرض کردیم.
    (PendingReward.objects.filter(id__in=picked_ids).update(withdrawn=True))

    return {"request_id": req_id, "picked_reward_ids": picked_ids, "amount": str(amount)}


def get_withdrawable_per_token(user):
    """
    خروجی: [{"token":"MGC","amount":"12.34500000"}, ...]
    فقط پاداش‌های claimed و withdrawn=False
    """
    if not user or not getattr(user, "id", None):
        return []
    rows = (
        PendingReward.objects.filter(user=user, status="claimed", withdrawn=False)
        .values("token__symbol")
        .annotate(total=Sum("amount"))
        .order_by("token__symbol")
    )
    out = []
    for r in rows:
        total = r["total"] or Decimal("0")
        out.append(
            {
                "token": r["token__symbol"],
                "amount": str(total.quantize(Decimal("0.00000000"))),
            }
        )
    return out


def claim_rewards(user):
    """
    1) استیک‌های فعال کاربر را تا 'الان' settle می‌کند؛
       اگر stake.pending_reward > 0 بود، آن را به PendingReward(status='claimed') منتقل می‌کند و صفر می‌کند.
    2) سپس همه‌ی PendingRewardهای 'pending' کاربر را هم به 'claimed' تغییر وضعیت می‌دهد.
    خروجی: مجموع کل، تعداد رکوردهای claim‌شده، تفکیک به ازای هر توکن، و زمان claim.
    """
    with transaction.atomic():
        now = timezone.now()

        # --- مرحله A: settle استیک‌های فعال و انتقال pending_reward به ledger (claimed) ---
        created_from_stakes = []
        # قفل خوشه‌ای روی استیک‌های فعال کاربر
        active_stakes = Stake.objects.select_for_update().filter(user=user, is_active=True)

        for st in active_stakes:
            # محاسبهٔ لحظه‌ای تا اکنون (اگر در ویو قبلاً صدا زده نشده باشد، اینجا محافظه‌کارانه دوباره می‌زنیم)
            accrue_until_now(st)

            amt = st.pending_reward or Decimal("0")
            if amt > 0:
                # فرض: هر Stake یک token دارد
                token = st.token
                pr = PendingReward.objects.create(
                    user=user,
                    token=token,
                    amount=amt,
                    status="claimed",  # مستقیم claimed می‌کنیم چون این مرحله claim است
                    claimed_at=now,
                )
                created_from_stakes.append(pr)

                # صفر کردن accumulator روی Stake
                st.pending_reward = Decimal("0")
                st.last_calculated_at = now
                st.save(update_fields=["pending_reward", "last_calculated_at"])

        # --- مرحله B: claim کردن PendingRewardهای قدیمیِ 'pending' ---
        qs = PendingReward.objects.select_for_update().filter(user=user, status="pending")

        # جمعِ دستهٔ pending قبل از update
        by_token_pending = list(qs.values(symbol=F("token__symbol")).annotate(total=Sum("amount")))
        updated = 0
        if by_token_pending:
            updated = qs.update(status="claimed", claimed_at=now)

        # --- مرحله C: تجمیع خروجی (دستهٔ جدید + دستهٔ قدیمی) ---
        agg = defaultdict(Decimal)

        # 1) دستهٔ قدیمی (که الان claimed شدند)
        for row in by_token_pending:
            sym = row["symbol"]
            tot = row["total"] or Decimal("0")
            agg[sym] += tot

        # 2) دستهٔ جدید (ساخته‌شده از stake.pending_reward)
        for pr in created_from_stakes:
            agg[pr.token.symbol] += pr.amount

        by_token = [{"symbol": sym, "total": amt} for sym, amt in agg.items()]
        claimed_total = sum(agg.values(), Decimal("0"))
        count = updated + len(created_from_stakes)

        # اگر هیچ‌چیزی claim نشد، خروجی صفر
        if count == 0:
            return {
                "claimed_total": Decimal("0"),
                "count": 0,
                "by_token": [],
                "claimed_at": None,
            }

        return {
            "claimed_total": claimed_total,
            "count": count,
            "by_token": by_token,  # [{symbol: 'RZ', total: Decimal(...)} , ...]
            "claimed_at": now,
        }


def _get_setting_min_withdrawals():
    # از settings اگر تعریف شده استفاده کن، وگرنه خالی
    return getattr(settings, "MIN_WITHDRAWALS", {})


def get_min_withdrawal_for_symbol(symbol: str) -> Decimal:
    """
    حداقل برداشت برای نماد (مثل 'RZ' یا 'MGC')
    اول از Config خوانده می‌شود:  key = MIN_WITHDRAW_<SYMBOL>
    اگر نبود، از settings.MIN_WITHDRAWALS می‌خوانیم.
    اگر هیچ‌کدام نبود، 0 برمی‌گردد.
    """
    sym = (symbol or "").upper().strip()
    if not sym:
        return Decimal("0")

    # 1) Config override
    cfg_key = f"MIN_WITHDRAW_{sym}"
    row = Config.objects.filter(key=cfg_key).order_by("-id").first()
    if row and getattr(row, "value", None):
        try:
            return Decimal(str(row.value))
        except (InvalidOperation, ValueError):
            pass  # اگر parse نشد، میریم سراغ settings

    # 2) settings fallback
    st = _get_setting_min_withdrawals()
    if isinstance(st, dict) and sym in st:
        try:
            return Decimal(str(st[sym]))
        except (InvalidOperation, ValueError):
            return Decimal("0")

    return Decimal("0")
