# apps/wallets/management/commands/process_withdraw_queue.py
from decimal import Decimal

from django.core.management.base import BaseCommand
from django.db import transaction
from django.utils import timezone

from apps.wallets.chain import (
    build_and_send_erc20_transfer,
    get_erc20_contract,
    get_web3,
    sanitize_receipt,
    to_wei,
)
from apps.wallets.constants import (
    BSC_CHAIN_ID,
    SERVER_WALLET_ADDRESS,
    SERVER_WALLET_PRIVATE_KEY,
    WAIT_RECEIPT_TIMEOUT,
)
from apps.wallets.models import PendingReward, WithdrawalItem, WithdrawRequest


class Command(BaseCommand):
    help = "Process queued withdraw requests and (mock) send tokens on-chain"

    def handle(self, *args, **options):
        processed = 0
        failed = 0

        # فقط درخواست‌های queued
        qs = WithdrawRequest.objects.filter(status="queued").order_by("id")[:50]

        if not qs.exists():
            self.stdout.write(self.style.WARNING("No queued withdraw requests."))
            return

        w3 = get_web3()

        for wr in qs:
            try:
                with transaction.atomic():
                    # انتخاب توکن غالب از WithdrawalItem ها (اولین آیتم)
                    witems = list(
                        WithdrawalItem.objects.select_for_update()
                        .filter(pending_reward__withdrawn=True)
                        .filter(  # فقط آیتم‌هایی که در ویو مصرف شده‌اند
                            # اگر WithdrawalItem فیلد FK به WithdrawRequest دارد، اینجا فیلتر کن:
                            # withdrawrequest=wr
                        )
                    )
                    if not witems:
                        # اگر لینک مستقیم نداریم، از PendingRewardهای مصرف‌شده‌ی کاربر کمک بگیر
                        # (fallback ساده: اولین pr claimed & withdrawn را بگیر)
                        pr = (
                            PendingReward.objects.filter(user=wr.user, withdrawn=True)
                            .order_by("-id")
                            .first()
                        )
                        token = getattr(pr, "token", None)
                    else:
                        pr = witems[0].pending_reward
                        token = getattr(pr, "token", None)

                    if not token:
                        raise ValueError("Cannot resolve token for this withdraw request.")

                    token_decimals = int(getattr(token, "decimals", 18))
                    token_contract_addr = getattr(token, "contract_address", None)

                    # آدرس مقصد را خودتان طبق بیزینس‌لاگیک انتخاب کنید:
                    # سناریوی ساده: آخرین اتصال کیف پول کاربر
                    from apps.wallets.models import WalletConnection

                    conn = WalletConnection.objects.filter(user=wr.user).order_by("-id").first()
                    if not conn or not conn.wallet_address:
                        raise ValueError("User has no connected wallet to send funds to.")

                    to_addr = conn.wallet_address

                    # مقدار به wei
                    amount_wei = to_wei(Decimal(wr.amount), token_decimals)

                    # قرارداد (mock یا واقعی)
                    contract = get_erc20_contract(
                        w3, token_contract_addr or "0x0", decimals=token_decimals
                    )

                    # در حالت mock این تابع بلافاصله tx_hash/receipt ساختگی می‌دهد
                    tx_hash, receipt = build_and_send_erc20_transfer(
                        token_contract=contract,
                        server_address=SERVER_WALLET_ADDRESS,
                        server_private_key=SERVER_WALLET_PRIVATE_KEY,
                        to_address=to_addr,
                        amount_wei=amount_wei,
                        nonce=None,  # در mock مهم نیست
                        gas_limit=None,
                        gas_price_wei=None,
                        chain_id=BSC_CHAIN_ID,
                        wait_timeout=WAIT_RECEIPT_TIMEOUT,
                    )

                    receipt_json = sanitize_receipt(receipt, w3)

                    # به‌روزرسانی وضعیت درخواست
                    wr.tx_hash = tx_hash
                    wr.details = {"receipt": receipt_json}
                    wr.status = "approved" if receipt_json.get("status", 0) == 1 else "failed"
                    wr.processed_at = timezone.now()
                    wr.save(update_fields=["tx_hash", "details", "status", "processed_at"])

                processed += 1
                self.stdout.write(
                    self.style.SUCCESS(f"Processed WR #{wr.id} → {wr.status} tx={wr.tx_hash}")
                )

            except Exception as e:
                failed += 1
                wr.status = "failed"
                wr.details = {"error": str(e)}
                wr.processed_at = timezone.now()
                wr.save(update_fields=["status", "details", "processed_at"])
                self.stderr.write(self.style.ERROR(f"Failed WR #{wr.id}: {e}"))

        self.stdout.write(self.style.SUCCESS(f"Done. processed={processed}, failed={failed}"))
