# apps/wallets/chain.py
import time
from decimal import Decimal

from django.utils.crypto import get_random_string

from .constants import CHAIN_MOCK_MODE, RPC_URL

# سعی می‌کنیم web3 را لود کنیم؛ اگر نبود یا mock روشن بود، میریم روی ساختگی
HAS_WEB3 = False
Web3 = None
HTTPProvider = None
try:
    if not CHAIN_MOCK_MODE:
        from web3 import HTTPProvider, Web3  # type: ignore

        HAS_WEB3 = True
except Exception:
    HAS_WEB3 = False
    Web3 = None
    HTTPProvider = None


# ---------- ابزارهای کمکی ----------
def to_wei(amount: Decimal, decimals: int) -> int:
    q = Decimal(10) ** decimals
    return int((Decimal(amount) * q).to_integral_value())


def from_wei(value: int, decimals: int) -> Decimal:
    q = Decimal(10) ** decimals
    return (Decimal(value) / q).quantize(Decimal("0.00000001"))


def sanitize_receipt(receipt: dict, w3=None) -> dict:
    # در حالت mock فقط همینی که داریم رو برمی‌گردونیم
    return {
        "status": receipt.get("status", 1),
        "transactionHash": receipt.get("transactionHash"),
        "gasUsed": receipt.get("gasUsed", 21000),
        "blockNumber": receipt.get("blockNumber", 12345678),
    }


# ---------- حالت MOCK ----------
class _MockEth:
    def __init__(self):
        self._gas_price = 1_000_000_000  # 1 Gwei ساختگی
        self._balance = 10**20  # یه بالانس ساختگی

    @property
    def gas_price(self):
        return self._gas_price

    def get_balance(self, addr):
        return self._balance

    def get_transaction_count(self, addr):
        # یه nonce ساختگی
        return 0


class _MockContractFunctions:
    def __init__(self, decimals=18):
        self._decimals = decimals

    def balanceOf(self, addr):
        class _Call:
            def call(self_inner):
                return 10**19  # 10 توکن ساختگی

        return _Call()

    def transfer(self, to, amount_wei):
        class _TransferObj:
            def estimateGas(self_inner, tx):
                return 120_000

        return _TransferObj()


class _MockContract:
    def __init__(self, decimals=18):
        self.functions = _MockContractFunctions(decimals=decimals)


class _MockWeb3:
    def __init__(self):
        self.eth = _MockEth()

    def toChecksumAddress(self, addr):
        return addr


def _fake_tx_hash():
    return "0x" + get_random_string(64, allowed_chars="0123456789abcdef")


def _fake_receipt():
    return {
        "status": 1,
        "transactionHash": _fake_tx_hash(),
        "gasUsed": 123456,
        "blockNumber": 9999999,
    }


# ---------- API موردنیاز ویو/ورکر ----------
def get_web3():
    """
    اگر CHAIN_MOCK_MODE=True یا web3 نصب/قابل استفاده نبود → mock
    در غیر این صورت وب۳ واقعی بر اساس RPC_URL.
    """
    if CHAIN_MOCK_MODE or not HAS_WEB3 or not RPC_URL:
        return _MockWeb3()
    return Web3(HTTPProvider(RPC_URL))


def get_erc20_contract(w3, contract_address: str, decimals: int = 18):
    """
    در حالت mock، فقط یک آبجکت ساختگی برمی‌گردانیم.
    در حالت واقعی باید با ABI لود شود.
    """
    if CHAIN_MOCK_MODE or not HAS_WEB3:
        return _MockContract(decimals=decimals)

    # حالت واقعی (بعداً کامل می‌کنیم) — الان raise تا یادت نره
    raise NotImplementedError(
        "Real get_erc20_contract is not implemented yet. Provide ABI & address."
    )


def build_and_send_erc20_transfer(
    token_contract,
    server_address: str,
    server_private_key: str,
    to_address: str,
    amount_wei: int,
    nonce=None,
    gas_limit: int = None,
    gas_price_wei: int = None,
    chain_id: int = None,
    wait_timeout: int = 30,
):
    """
    در حالت mock، یک tx_hash ساختگی و receipt موفق برمی‌گردانیم.
    """
    if CHAIN_MOCK_MODE or not HAS_WEB3:
        # شبیه‌سازی کمی تأخیر شبکه
        time.sleep(0.2)
        fake_hash = _fake_tx_hash()
        receipt = _fake_receipt()
        receipt["transactionHash"] = fake_hash
        return fake_hash, receipt

    # حالت واقعی (بعداً کامل می‌کنیم)
    raise NotImplementedError("Real build_and_send_erc20_transfer is not implemented yet.")
