# ── Standard Library
import secrets
import uuid
from datetime import timedelta
from decimal import Decimal

import httpx
from drf_yasg import openapi
from drf_yasg.utils import swagger_auto_schema
from eth_account import Account
from eth_account.messages import encode_defunct
from rest_framework import status
from rest_framework.permissions import AllowAny, IsAdminUser, IsAuthenticated
from rest_framework.response import Response
from rest_framework.throttling import ScopedRateThrottle
from rest_framework.views import APIView
from rest_framework_simplejwt.tokens import RefreshToken

# ── Django
from django.conf import settings
from django.contrib.auth import get_user_model

# ── Third-Party
from django.db import IntegrityError, transaction
from django.db.models import Count, Max, Q, Sum
from django.shortcuts import get_object_or_404
from django.utils import timezone

# ── Project: Badges / Core
from apps.badges.models import UserBadge
from apps.badges.services import user_badge_bonus_bp
from apps.core.utils.jwt import create_access_token  # اگر JWT اینجاست؛ مسیر واقعی‌ات را بگذار
from apps.core.utils.percentages import compute_effective_percent
from apps.miners.models import Miner  # اگر مدل Miner اینجاست
from apps.stakes.models import Stake

# ── Project: Users
from apps.users.models import (
    EmailVerification,
    ReferralBonusConfig,
    ReferralInvite,
    ReferralRelation,
    User,
    UserProfile,
)
from apps.users.serializers import (
    EmailChangeConfirmSerializer,
    EmailChangeRequestSerializer,
    EmailRequestSerializer,
    LoginSerializer,
    ProfileUpdateSerializer,
    ReferralInviteCreateSerializer,
    ReferralInviteResponseSerializer,
    ReferralInviteSerializer,
    ReferralMyCodeSerializer,
    ReferralResolveResponseSerializer,
    ReferralRotateResponseSerializer,
    ReferralStatsSerializer,
    ReferralUserSerializer,
    ResetPasswordSerializer,
    VerifyOTPRequestSerializer,
    WalletLoginSerializer,
    WalletSignupSerializer,
)
from apps.users.services.referrals import (
    CFG_PER_ACTIVE_BONUS_BP,
    _get_config_int,
    count_active_referrals,
)

# ── Project: Wallets / Stakes / Tokens
from apps.wallets.models import (
    PendingReward,
    Wallet,
    WalletAuthNonce,
    WalletConnection,
    WalletTransaction,
)

from .serializers import GoogleAuthSerializer, UserResponseSerializer  # اگر این‌ها در همین اپ هستند

# ── Project: Utils
from .utils import send_reset_password_email  # اگر ایمیل‌زن اینجاست
from .utils import create_otp_for_email, verify_otp
from apps.events.services import register_login_activity

def update_referral_bonus(user):
    bonus_config = ReferralBonusConfig.objects.first()
    if bonus_config:
        user.referral_bonus += bonus_config.bonus_percentage
        user.save()


def generate_reset_token(length=16):
    return secrets.token_urlsafe(length)[:16]  # تولید توکن 16 کاراکتری امن


def create_reset_token_for_email(email, token_type="reset_password"):
    # حذف توکن‌های منقضی شده
    EmailVerification.objects.filter(
        email=email, is_used=False, expires_at__lt=timezone.now()
    ).delete()

    token = generate_reset_token()
    expiry = timezone.now() + timedelta(hours=1)

    reset_token = EmailVerification.objects.create(
        email=email, otp_code=token, expires_at=expiry, otp_type=token_type
    )

    reset_link = f"https://coinmaining.game/reset-password?token={token}"
    send_reset_password_email(email, reset_link)

    return reset_token


def verify_reset_token(token, token_type="reset_password"):
    token_obj = EmailVerification.objects.filter(
        otp_code=token, is_used=False, expires_at__gt=timezone.now(), otp_type=token_type
    ).first()

    if token_obj:
        token_obj.is_used = True
        token_obj.save()
        return token_obj.email
    return None


def _invite_link(code: str) -> str:
    base = getattr(settings, "REFERRAL_LINK_BASE", "https://app.example.com/invite")
    sep = "&" if "?" in base else "?"
    return f"{base}{sep}code={code}"


def _gen_ref_code():
    # کد کوتاه، URL-safe و نسبتاً خوانا
    return secrets.token_urlsafe(6).replace("_", "").replace("-", "").upper()


class SendEmailCodeView(APIView):
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "auth"

    @swagger_auto_schema(request_body=EmailRequestSerializer)
    def post(self, request):
        return Response(
            {"detail": "Signup is wallet-only. Please use /api/users/auth/wallet_signup/"},
            status=status.HTTP_403_FORBIDDEN,
        )


class VerifyCodeView(APIView):
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "auth"

    @swagger_auto_schema(request_body=VerifyOTPRequestSerializer)
    def post(self, request):
        return Response(
            {"detail": "Signup is wallet-only. Please use /api/users/auth/wallet_signup/"},
            status=status.HTTP_403_FORBIDDEN,
        )


class SetUsernameView(APIView):
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "auth"

    @swagger_auto_schema(
        manual_parameters=[],
        request_body=openapi.Schema(
            type=openapi.TYPE_OBJECT,
            required=["email", "username"],
            properties={
                "email": openapi.Schema(type=openapi.TYPE_STRING, format="email"),
                "username": openapi.Schema(type=openapi.TYPE_STRING),
                "password": openapi.Schema(type=openapi.TYPE_STRING, minLength=6),
                "referred_by": openapi.Schema(type=openapi.TYPE_STRING),
            },
        ),
        consumes=["multipart/form-data"],
    )
    def post(self, request):
        return Response(
            {"detail": "Signup is wallet-only. Please use /api/users/auth/wallet_signup/"},
            status=status.HTTP_403_FORBIDDEN,
        )


class EmailPasswordLoginView(APIView):
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "auth"

    @swagger_auto_schema(request_body=LoginSerializer)
    def post(self, request):
        serializer = LoginSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        email = serializer.validated_data["email"]
        password = serializer.validated_data["password"]

        user = User.objects.filter(email=email).first()
        if not user:
            return Response({"detail": "User not found."}, status=404)

        if not user.has_usable_password():
            return Response(
                {"detail": "You haven't set your password yet. Please complete registration."},
                status=403,
            )

        if not user.check_password(password):
            return Response({"detail": "Incorrect password."}, status=401)

        if not user.is_verified:
            return Response({"detail": "Please verify your email first."}, status=403)

        if not user.is_active:
            return Response({"detail": "Your account is inactive."}, status=403)
        register_login_activity(user)
        tokens = user.get_tokens()
        return Response(
            {
                "access_token": tokens["access"],
                "refresh_token": tokens["refresh"],
                "token_type": "bearer",
                "user": UserResponseSerializer(user).data,
            }
        )


class RequestPasswordResetView(APIView):
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "auth"

    @swagger_auto_schema(
        request_body=EmailRequestSerializer,
        responses={
            200: "Password reset link sent to your email.",
            404: "User not found.",
            429: "Reset link recently sent. Please wait a moment before requesting again.",
        },
    )
    def post(self, request):
        serializer = EmailRequestSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        email = serializer.validated_data["email"]

        user = User.objects.filter(email=email).first()
        if not user:
            return Response({"detail": "User not found."}, status=404)

        # جلوگیری از اسپم درخواست لینک
        recent_token = EmailVerification.objects.filter(
            email=email,
            otp_type="reset_password",
            expires_at__gt=timezone.now() - timedelta(seconds=120),
        ).first()
        if recent_token:
            return Response(
                {
                    "detail": "Reset link recently sent. Please wait a moment before requesting again."
                },
                status=429,
            )

        create_reset_token_for_email(email, token_type="reset_password")

        return Response({"detail": "Password reset link sent to your email."}, status=200)


class ResetPasswordView(APIView):
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "auth"

    @swagger_auto_schema(
        request_body=ResetPasswordSerializer,
        responses={
            200: "Password reset successfully.",
            400: "Invalid or expired reset token.",
            404: "User not found.",
        },
    )
    def post(self, request):
        serializer = ResetPasswordSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        token = serializer.validated_data["token"]
        new_password = serializer.validated_data["password"]

        # تأیید توکن و دریافت ایمیل کاربر
        email = verify_reset_token(token, token_type="reset_password")
        if not email:
            return Response({"detail": "Invalid or expired reset token."}, status=400)

        user = User.objects.filter(email=email).first()
        if not user:
            return Response({"detail": "User not found."}, status=404)

        user.set_password(new_password)
        user.save()

        return Response({"detail": "Password reset successfully."}, status=200)


class WalletSignupView(APIView):
    authentication_classes = []  # anonymous allowed
    permission_classes = [AllowAny]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "signup"

    signup_responses = {
        201: openapi.Response("signup_ok"),
        200: openapi.Response("wallet_already_connected"),
        400: openapi.Response("bad request"),
        409: openapi.Response("wallet_taken"),
    }

    @swagger_auto_schema(
        operation_summary="Wallet-only signup (email, username, avatar_key, signature + nonce_id required)",
        operation_description=(
            "Flow:\n"
            "1) POST /api/wallets/wallet/metamask/nonce/ → get {id, message}\n"
            "2) Sign `message` with wallet → signature\n"
            "3) POST here with: address, signature, nonce_id, provider, email, username, avatar_key, [invite]"
        ),
        request_body=WalletSignupSerializer,
        responses=signup_responses,
    )
    @transaction.atomic
    def post(self, request, *args, **kwargs):
        # 0) validate body
        try:
            ser = WalletSignupSerializer(data=request.data)
            ser.is_valid(raise_exception=True)

            address = ser.validated_data["address"].strip()
            signature = ser.validated_data["signature"].strip()
            nonce_id = ser.validated_data["nonce_id"]
            provider = (ser.validated_data.get("provider") or "metamask").strip().lower()
            email = ser.validated_data["email"].strip().lower()
            username = ser.validated_data["username"].strip()
            avatar_key = ser.validated_data["avatar_key"]

            # 1) lock & check nonce, verify signature, connect/create user, consume invite — all atomically
            with transaction.atomic():
                nonce = WalletAuthNonce.objects.select_for_update().filter(id=nonce_id).first()
                if not nonce:
                    return Response(
                        {"detail": "Invalid nonce_id."}, status=status.HTTP_400_BAD_REQUEST
                    )

                # provider/address consistency (if model has fields)
                n_provider = getattr(nonce, "provider", None)
                if n_provider and n_provider != provider:
                    return Response(
                        {"detail": "nonce/provider mismatch"}, status=status.HTTP_400_BAD_REQUEST
                    )

                n_address = getattr(nonce, "address", None)
                if n_address and n_address.lower() != address.lower():
                    return Response(
                        {"detail": "nonce/address mismatch"}, status=status.HTTP_400_BAD_REQUEST
                    )

                if getattr(nonce, "is_used", False):
                    return Response(
                        {"detail": "nonce already used"}, status=status.HTTP_400_BAD_REQUEST
                    )
                n_expires = getattr(nonce, "expires_at", None)
                if n_expires and timezone.now() > n_expires:
                    return Response({"detail": "nonce expired"}, status=status.HTTP_400_BAD_REQUEST)

                # message to verify
                if getattr(nonce, "message", None):
                    msg_text = nonce.message
                elif getattr(nonce, "nonce", None):
                    msg_text = f"Login with nonce: {nonce.nonce}"
                else:
                    return Response(
                        {"detail": "nonce has no message to sign"},
                        status=status.HTTP_400_BAD_REQUEST,
                    )

                try:
                    msg = encode_defunct(text=msg_text)
                    recovered = Account.recover_message(msg, signature=signature)
                except Exception as e:
                    return Response(
                        {"detail": f"signature verification failed: {e}"},
                        status=status.HTTP_400_BAD_REQUEST,
                    )

                if recovered.lower() != address.lower():
                    return Response(
                        {"detail": "signature does not match address"},
                        status=status.HTTP_400_BAD_REQUEST,
                    )

                # 2) wallet already connected?
                existing_conn = (
                    WalletConnection.objects.select_for_update()
                    .filter(wallet_address__iexact=address)
                    .first()
                )
                if existing_conn and existing_conn.user_id:
                    # same user → treat as login
                    if existing_conn.user and existing_conn.user.email.lower() == email:
                        # mark nonce used (audit fields if exist)
                        nf = []
                        if hasattr(nonce, "is_used"):
                            nonce.is_used = True
                            nf.append("is_used")
                        if hasattr(nonce, "used_at"):
                            nonce.used_at = timezone.now()
                            nf.append("used_at")
                        if hasattr(nonce, "used_by") and not getattr(nonce, "used_by_id", None):
                            nonce.used_by = existing_conn.user
                            nf.append("used_by")
                        if hasattr(nonce, "user") and not getattr(nonce, "user_id", None):
                            nonce.user = existing_conn.user
                            nf.append("user")
                        nonce.save(update_fields=nf) if nf else nonce.save()

                        refresh = RefreshToken.for_user(existing_conn.user)
                        return Response(
                            {
                                "detail": "wallet_already_connected",
                                "user_id": existing_conn.user_id,
                                "email": existing_conn.user.email,
                                "access": str(refresh.access_token),
                                "refresh": str(refresh),
                            },
                            status=status.HTTP_200_OK,
                        )

                    # connected to another user
                    return Response(
                        {"detail": "This wallet is already connected to another user."},
                        status=status.HTTP_409_CONFLICT,
                    )

                # 3) create/update user by email
                user = User.objects.select_for_update().filter(email__iexact=email).first()
                if user:
                    if not getattr(user, "username", None):
                        user.username = username
                        user.save(update_fields=["username"])
                else:
                    user = User.objects.create(
                        email=email,
                        username=username,
                        is_active=True,
                    )

                # 4) ensure profile + avatar
                profile, _ = UserProfile.objects.get_or_create(user=user)
                pf = []
                if hasattr(profile, "avatar_key"):
                    if getattr(profile, "avatar_key", None) != avatar_key:
                        profile.avatar_key = avatar_key
                        pf.append("avatar_key")
                elif hasattr(profile, "avatar"):
                    if getattr(profile, "avatar", None) != avatar_key:
                        profile.avatar = avatar_key
                        pf.append("avatar")
                if pf:
                    profile.save(update_fields=pf)

                # 5) connect wallet (verify if field exists)
                if existing_conn and not existing_conn.user_id:
                    existing_conn.user = user
                    existing_conn.provider = provider
                    fields = ["user", "provider"]
                    if hasattr(existing_conn, "is_verified"):
                        existing_conn.is_verified = True
                        fields.append("is_verified")
                    existing_conn.save(update_fields=fields)
                else:
                    wc = WalletConnection(user=user, wallet_address=address, provider=provider)
                    if hasattr(wc, "is_verified"):
                        wc.is_verified = True
                    wc.save()

                # 6) invite code (safe & idempotent)
                # --- begin: invite linking & used_count ---

                with transaction.atomic():
                    inv = ser.validated_data.get("_invite_obj")  # از validate گرفتیم
                    if inv:
                        # رابطه دعوت را بساز اگر قبلاً وجود ندارد
                        rel, created = ReferralRelation.objects.get_or_create(
                            inviter=inv.inviter,
                            invitee=user,
                            defaults={"active": False},
                        )

                        # تعیین active بر اساس وجود حداقل یک stake فعال
                        has_active = Stake.objects.filter(user=user, is_active=True).exists()
                        if rel.active != has_active:
                            rel.active = has_active
                            rel.save(update_fields=["active"])

                        # مصرف invite
                        used = (inv.used_count or 0) + 1
                        inv.used_count = used
                        # اگر ظرفیت داشت و پر شد، غیرفعال کن
                        if inv.max_uses is not None and used >= inv.max_uses:
                            inv.is_active = False
                        inv.save(update_fields=["used_count", "is_active"])
                # --- end: invite linking & used_count ---
                # 7) mark nonce used (audit)
                nf = []
                if hasattr(nonce, "is_used"):
                    nonce.is_used = True
                    nf.append("is_used")
                if hasattr(nonce, "used_at"):
                    nonce.used_at = timezone.now()
                    nf.append("used_at")
                if hasattr(nonce, "used_by") and not getattr(nonce, "used_by_id", None):
                    nonce.used_by = user
                    nf.append("used_by")
                if hasattr(nonce, "user") and not getattr(nonce, "user_id", None):
                    nonce.user = user
                    nf.append("user")
                nonce.save(update_fields=nf) if nf else nonce.save()

            # 8) issue tokens (outside transaction)
            refresh = RefreshToken.for_user(user)
            return Response(
                {
                    "detail": "signup_ok",
                    "user_id": user.id,
                    "email": user.email,
                    "access": str(refresh.access_token),
                    "refresh": str(refresh),
                    "referral": {
                        "invite_used": bool(ser.validated_data.get("_invite_obj")),
                        "inviter_id": (
                            ser.validated_data.get("_invite_obj").inviter_id
                            if ser.validated_data.get("_invite_obj")
                            else None
                        ),
                    },
                },
                status=status.HTTP_201_CREATED,
            )
        except Exception as e:
            import sys
            import traceback

            traceback.print_exc(file=sys.stdout)
            return Response({"detail": f"DEBUG ERROR: {str(e)}"}, status=500)


class GoogleLoginView(APIView):
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "auth"

    async def post(self, request):
        serializer = GoogleAuthSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        token = serializer.validated_data["access_token"]

        async with httpx.AsyncClient() as client:
            response = await client.get(
                "https://www.googleapis.com/oauth2/v2/userinfo",
                headers={"Authorization": f"Bearer {token}"},
            )

        if response.status_code != 200:
            return Response({"detail": "توکن گوگل معتبر نیست"}, status=401)

        data = response.json()
        email = data.get("email")

        if not email:
            return Response({"detail": "ایمیل یافت نشد"}, status=400)

        user = User.objects.filter(email=email).first()
        if not user:
            user = User.objects.create(
                email=email,
                username=email.split("@")[0] + str(uuid.uuid4())[:4],
                referral_code=str(uuid.uuid4())[:8],
                is_verified=True,
            )
            user.set_password("")
            user.save()
        register_login_activity(user)

        token = create_access_token({"sub": user.email})
        return Response({"access_token": token, "token_type": "bearer"})


class LogoutView(APIView):

    def post(self, request):
        return Response({"detail": "Logout successful"}, status=200)


class MeView(APIView):
    permission_classes = [IsAuthenticated]

    def get(self, request, *args, **kwargs):
        u = request.user
        p = getattr(u, "profile", None)

        # آخرین اتصال ولت
        conn = WalletConnection.objects.filter(user=u).order_by("-id").first()
        wallet_connected = bool(conn)
        wallet_address = getattr(conn, "wallet_address", None) if conn else None
        wallet_provider = getattr(conn, "provider", None) if conn else None

        # استیک‌های فعال (+ درصد مؤثر)
        active_stakes = (
            Stake.objects.filter(user=u, is_active=True)
            .select_related("token", "miner__plan")
            .order_by("-created_at")
        )
        stakes_payload = []
        for st in active_stakes:
            token_symbol = getattr(st.token, "symbol", None)
            plan = getattr(getattr(st, "miner", None), "plan", None)
            base_percent = getattr(plan, "monthly_reward_percent", None) or Decimal("4.5")

            # اگر compute_effective_percent در دسترس نبود، فال‌بک ساده:
            try:
                eff_percent = compute_effective_percent(u, base_percent)
            except Exception:
                eff_percent = base_percent

            video_url = getattr(plan, "video_url", None)
            started_at = getattr(st, "created_at", None)
            stakes_payload.append(
                {
                    "token": token_symbol,
                    "amount": str(getattr(st, "amount", "0")),
                    "plan": getattr(plan, "name", None),
                    "base_percent": str(base_percent),
                    "effective_percent": str(eff_percent),
                    "video_url": video_url,
                    "started_at": started_at.isoformat() if started_at else None,
                }
            )

        # withdrawable: مجموع کل + به تفکیک توکن (claimed و هنوز withdraw نشده)
        pr_qs_claimed = PendingReward.objects.filter(user=u, status="claimed", withdrawn=False)
        withdrawable_total = pr_qs_claimed.aggregate(total=Sum("amount"))["total"] or Decimal("0")
        per_token = (
            pr_qs_claimed.values("token__symbol")
            .annotate(total=Sum("amount"))
            .order_by("token__symbol")
        )
        withdrawable_per_token = {
            row["token__symbol"]: str(row["total"]) for row in per_token if row["token__symbol"]
        }

        # تراکنش‌های اخیر (ایمن نسبت به نبودن فیلدها)
        recent_txs = []
        wallet = Wallet.objects.filter(user=u).first()
        if wallet:
            for tx in WalletTransaction.objects.filter(wallet=wallet).order_by("-id")[:10]:
                created_at = getattr(tx, "created_at", None)
                recent_txs.append(
                    {
                        "id": tx.id,
                        "amount": str(getattr(tx, "amount", "")),
                        "type": getattr(
                            tx, "tx_type", None
                        ),  # اگر فیلدی با این نام ندارید، حذف/تطبیق دهید
                        "created_at": created_at.isoformat() if created_at else None,
                    }
                )

        # توضیحات امتیاز رفرال و بج‌ها
        try:
            ref_per_active_bp = _get_config_int(CFG_PER_ACTIVE_BONUS_BP, 1)
        except Exception:
            ref_per_active_bp = 1

        data = {
            "user": {
                "id": u.id,
                "email": getattr(u, "email", None),
                "username": getattr(u, "username", None),
                "display_name": getattr(p, "display_name", None),
                "avatar": getattr(p, "avatar", None) or getattr(p, "avatar_key", None),
            },
            "wallet": {
                "connected": wallet_connected,
                "provider": wallet_provider,
                "address": wallet_address,
            },
            "stakes": stakes_payload,
            "withdrawable_total": str(withdrawable_total),
            "withdrawable_per_token": withdrawable_per_token,
            "recent_transactions": recent_txs,
        }

        data["explainers"] = {
            "referral": {
                "active_referrals": count_active_referrals(u),
                "per_active_bp": ref_per_active_bp,
            },
            "badges": {"total_badge_bonus_bp": user_badge_bonus_bp(u)},
        }

        # badges — با ترتیب درست: display_order
        user_badges = (
            UserBadge.objects.select_related("badge")
            .filter(user=u, is_active=True)  # اگر فیلد is_active روی UserBadge دارید
            .order_by("badge__display_order", "badge__id")  # ← اصلاح کلیدی
        )

        badges_payload = []
        for ub in user_badges:
            b = ub.badge
            awarded_at = getattr(ub, "awarded_at", None)
            badges_payload.append(
                {
                    "key": getattr(b, "key", None),
                    "name": getattr(b, "name", None),
                    "type": getattr(b, "type", None),
                    "reward_rate_bp": getattr(b, "reward_rate_bp", 0),
                    "awarded_at": awarded_at.isoformat() if awarded_at else None,
                }
            )

        data["badges"] = badges_payload
        return Response(data)


class ProfileUpdateView(APIView):
    permission_classes = [IsAuthenticated]

    def patch(self, request, *args, **kwargs):
        ser = ProfileUpdateSerializer(data=request.data, partial=True, context={"request": request})
        ser.is_valid(raise_exception=True)

        u = request.user
        p = getattr(u, "profile", None) or UserProfile.objects.create(user=u)

        data = ser.validated_data
        user_updates = []
        profile_updates = []

        try:
            with transaction.atomic():
                # user fields
                if "username" in data and data["username"] and u.username != data["username"]:
                    u.username = data["username"]
                    user_updates.append("username")

                if "email" in data and data["email"] and u.email != data["email"]:
                    u.email = data["email"]
                    user_updates.append("email")

                if user_updates:
                    u.save(update_fields=user_updates)

                # profile fields
                if "display_name" in data and p.display_name != data["display_name"]:
                    p.display_name = data["display_name"]
                    profile_updates.append("display_name")

                if "avatar_key" in data and data["avatar_key"]:
                    if getattr(p, "avatar_key", None) != data["avatar_key"]:
                        p.avatar_key = data["avatar_key"]
                        profile_updates.append("avatar_key")

                if profile_updates:
                    p.save(update_fields=profile_updates)

        except IntegrityError as e:
            return Response({"detail": "conflict", "error": str(e)}, status=409)

        return Response(
            {
                "detail": "profile_updated",
                "user": {
                    "id": u.id,
                    "email": u.email,
                    "username": u.username,
                },
                "profile": {
                    "display_name": getattr(p, "display_name", None),
                    "avatar_key": getattr(p, "avatar_key", None),
                },
            },
            status=200,
        )


class MyReferralCodeView(APIView):
    permission_classes = [IsAuthenticated]

    def get(self, request):
        if not request.user.referral_code:
            return Response({"detail": "Referral code not generated"}, status=500)

        # گرفتن درصد سود از تنظیمات
        config = ReferralBonusConfig.objects.first()
        bonus_percentage = float(config.bonus_percentage) if config else 0.0

        return Response(
            {"referral_code": request.user.referral_code, "bonus_percentage": bonus_percentage}
        )


class MyReferralsListView(APIView):
    permission_classes = [IsAuthenticated]

    def get(self, request):
        referred_users = User.objects.filter(referred_by=request.user.referral_code)
        serializer = ReferralUserSerializer(referred_users, many=True)
        referred_count = referred_users.count()
        mining_boost = (
            Miner.objects.filter(user__in=referred_users).aggregate(total=Sum("power"))["total"]
            or 0
        )

        # مقدار درصد سود دعوت از تنظیمات
        config = ReferralBonusConfig.objects.first()
        bonus_percentage = float(config.bonus_percentage) if config else 0.0

        return Response(
            {
                "referrals": serializer.data,
                "referred_count": referred_count,
                "mining_boost": mining_boost,
                "bonus_percentage": bonus_percentage,  # 🎯 اضافه شد
            }
        )


class ReferralPowerBonusView(APIView):
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "claim"
    permission_classes = [IsAdminUser]

    def get(self, request):
        config = ReferralBonusConfig.objects.first()
        if config:
            return Response({"power_bonus_percentage": config.bonus_percentage})
        return Response(
            {"detail": "Power bonus percentage not set"}, status=status.HTTP_404_NOT_FOUND
        )

    def put(self, request):
        power_bonus = request.data.get("power_bonus_percentage")
        if not power_bonus:
            return Response(
                {"detail": "Power bonus percentage is required"}, status=status.HTTP_400_BAD_REQUEST
            )

        try:
            power_bonus = float(power_bonus)
        except ValueError:
            return Response(
                {"detail": "Invalid percentage value"}, status=status.HTTP_400_BAD_REQUEST
            )

        # ذخیره درصد جدید
        config, created = ReferralBonusConfig.objects.get_or_create(key="referral_power_bonus")
        config.bonus_percentage = power_bonus
        config.save()

        return Response({"detail": "Power bonus updated successfully"})


class WalletLoginView(APIView):
    """
    POST /api/users/auth/wallet_login/
    لاگین با امضای پیام nonce (بدون ساخت کاربر جدید).
    اگر والت به کاربری وصل نباشد، 404 می‌دهد (یوزر باید قبلاً signup کرده باشد).
    """

    authentication_classes = []
    permission_classes = [AllowAny]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "auth"  # مثل nonce/signup

    login_responses = {
        200: openapi.Response("login_ok"),
        400: openapi.Response("bad request"),
        404: openapi.Response("wallet_not_linked"),
    }

    @swagger_auto_schema(
        operation_summary="Wallet login using signed nonce (no user creation)",
        operation_description=(
            "Flow: 1) POST /wallet/metamask/nonce/ to get nonce_id & message "
            "2) Sign the message with the wallet "
            "3) POST here with address, signature, nonce_id, provider"
        ),
        request_body=WalletLoginSerializer,  # address, signature, nonce_id, provider
        responses=login_responses,
    )
    def post(self, request, *args, **kwargs):
        ser = WalletLoginSerializer(data=request.data)
        ser.is_valid(raise_exception=True)

        address = ser.validated_data["address"].strip()
        signature = ser.validated_data["signature"].strip()
        nonce_id = ser.validated_data["nonce_id"]
        provider = ser.validated_data["provider"]

        if not nonce_id:
            return Response({"detail": "nonce_id is required"}, status=400)

        # 1) نانس را پیدا و اعتبارسنجی اولیه
        na = get_object_or_404(WalletAuthNonce, pk=nonce_id)

        if getattr(na, "is_used", False):
            return Response({"detail": "nonce already used"}, status=400)
        if getattr(na, "expires_at", None) and timezone.now() > na.expires_at:
            return Response({"detail": "nonce expired"}, status=400)
        if na.address.lower() != address.lower():
            return Response({"detail": "nonce/address mismatch"}, status=400)
        if (na.provider or "metamask") != provider:
            return Response({"detail": "nonce/provider mismatch"}, status=400)

        # پیام برای verify (ترجیح با message؛ اگر نبود از nonce خام)
        if getattr(na, "message", None):
            msg_text = na.message
        elif getattr(na, "nonce", None):
            msg_text = f"Login with nonce: {na.nonce}"
        else:
            return Response({"detail": "nonce has no message to sign"}, status=400)

        # 2) صحت امضا
        try:
            msg = encode_defunct(text=msg_text)
            recovered = Account.recover_message(msg, signature=signature)
        except Exception as e:
            return Response({"detail": f"signature verification failed: {e}"}, status=400)
        if recovered.lower() != address.lower():
            return Response({"detail": "signature does not match address"}, status=400)

        # 3) اتصال والت باید از قبل به کاربری وصل باشد
        conn = (
            WalletConnection.objects.filter(wallet_address__iexact=address)
            .select_related("user")
            .first()
        )
        if not conn or not conn.user_id:
            return Response({"detail": "wallet not linked"}, status=404)
        user = conn.user

        # (اختیاری) provider / is_verified به‌روز شود
        fields = []
        if hasattr(conn, "provider") and conn.provider != provider:
            conn.provider = provider
            fields.append("provider")
        if hasattr(conn, "is_verified") and not conn.is_verified:
            conn.is_verified = True
            fields.append("is_verified")
        if fields:
            conn.save(update_fields=fields)

        # 4) مصرف نانس بعد از موفقیت
        with transaction.atomic():
            if hasattr(na, "is_used"):
                na.is_used = True
            if hasattr(na, "used_at"):
                na.used_at = timezone.now()
            # اگر مدل نانس فیلدی مثل used_by یا user دارد، با صاحب واقعی ست کن (نه request.user)
            if hasattr(na, "used_by_id"):
                na.used_by = user
            elif hasattr(na, "user_id"):
                na.user = user
            na.save()
        refresh = RefreshToken.for_user(user)
        register_login_activity(user)
        return Response(
            {
                "detail": "login_ok",
                "user_id": user.id,
                "email": getattr(user, "email", None),
                "access": str(refresh.access_token),
                "refresh": str(refresh),
            },
            status=200,
        )


class ReferralInviteCreateView(APIView):
    permission_classes = [IsAuthenticated]

    @swagger_auto_schema(
        operation_summary="Create referral invite code",
        request_body=ReferralInviteCreateSerializer,
        responses={200: ReferralInviteSerializer},
    )
    def post(self, request, *args, **kwargs):
        ser = ReferralInviteCreateSerializer(data=request.data)
        ser.is_valid(raise_exception=True)

        max_uses = ser.validated_data.get("max_uses")
        ttl_hours = ser.validated_data.get("ttl_hours")

        inv = ReferralInvite.create_for(request.user, max_uses=max_uses, ttl_hours=ttl_hours)

        base = getattr(settings, "FRONTEND_BASE_URL", "https://app.example.com")
        url = f"{base}/signup?invite={inv.code}"

        out = ReferralInviteSerializer(
            {
                "code": inv.code,
                "url": url,
                "is_active": inv.is_active,
                "max_uses": inv.max_uses,
                "used_count": inv.used_count,
                "expires_at": inv.expires_at,
            }
        ).data
        return Response(out, status=200)

    @swagger_auto_schema(responses={200: ReferralInviteSerializer(many=True)})
    def get(self, request):
        invites = ReferralInvite.objects.filter(inviter=request.user).order_by("-created_at")
        data = ReferralInviteSerializer(invites, many=True).data
        return Response({"invites": data})


class ReferralInviteView(APIView):
    """
    GET /api/users/referral/invites/
    اگر برای کاربر قبلاً کدی وجود دارد همان را می‌دهد؛ وگرنه می‌سازد.
    """

    permission_classes = [IsAuthenticated]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "auth"

    @swagger_auto_schema(
        operation_summary="Get or create my referral invite link",
        responses={200: ReferralInviteResponseSerializer},
    )
    def get(self, request):
        inv = ReferralInvite.objects.filter(inviter=request.user).order_by("-id").first()
        if not inv:
            inv = ReferralInvite.create_for(
                inviter=request.user
            )  # این متد را قبلاً در شل استفاده کردی
        data = {
            "code": inv.code,
            "link": _invite_link(inv.code),
        }
        return Response(data, status=200)


class ReferralInviteRotateView(APIView):
    """
    POST /api/users/referral/invites/rotate/
    یک کد جدید می‌سازد (قدیمی پابرجاست یا باطل می‌شود؛ بستگی به مدل‌ شما دارد).
    """

    permission_classes = [IsAuthenticated]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "auth"

    @swagger_auto_schema(
        operation_summary="Rotate my referral invite code",
        responses={200: ReferralRotateResponseSerializer},
    )
    def post(self, request):
        old = ReferralInvite.objects.filter(inviter=request.user).order_by("-id").first()
        old_code = getattr(old, "code", None)

        new_inv = ReferralInvite.create_for(inviter=request.user)
        data = {
            "old_code": old_code or "",
            "new_code": new_inv.code,
            "link": _invite_link(new_inv.code),
        }
        return Response(data, status=200)


class ReferralStatsView(APIView):
    permission_classes = [IsAuthenticated]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "auth"

    @swagger_auto_schema(
        operation_summary="Referral stats (total / active / last invited at)",
        responses={200: ReferralStatsSerializer},
    )
    def get(self, request):
        qs = ReferralRelation.objects.filter(inviter=request.user)
        agg = qs.aggregate(
            total=Count("id"),
            active=Count("id", filter=Q(active=True)),
            last=Max("created_at"),
        )

        data = {
            "total_referrals": agg.get("total") or 0,
            "active_referrals": agg.get("active") or 0,
            "last_invited_at": agg.get("last"),  # DRF خودش ISO می‌کند
        }

        # هر دو درست‌اند؛ یکی را انتخاب کن:
        # return Response(data, status=200)
        ser = ReferralStatsSerializer(instance=data)
        return Response(ser.data, status=200)


def _avatar_url_from_profile(request, user):
    prof = getattr(user, "profile", None)
    if not prof:
        return ""

    # اگر کلید آواتار نباشه، با pr-<id> بساز
    key = getattr(prof, "avatar_key", "") or f"pr-{user.id}"

    # پسوند رو از settings بردار، پیش‌فرض jpeg
    ext = getattr(settings, "AVATAR_EXT", ".jpeg")
    if not ext.startswith("."):
        ext = f".{ext}"

    base = (getattr(settings, "AVATAR_BASE_URL", "") or "").rstrip("/")
    if base:
        # خروجی دقیقاً مثل چیزی که مدیر می‌خواد:
        # https://coinmaining.game/profiles/pr-125.jpeg
        return f"{base}/{key}{ext}"

    # allback (اگر BASE ست نبود)، با دامنه فعلی API بسازیم
    return request.build_absolute_uri(f"/profiles/{key}{ext}")


DEFAULT_PER_ACTIVE_BONUS_BP = Decimal("1")  # پیش‌فرض 10bp = 0.10%
DEFAULT_PER_ACTIVE_BONUS_FRACTION = DEFAULT_PER_ACTIVE_BONUS_BP / Decimal(10000)


def _safe_per_active_bonus_decimal() -> Decimal:
    try:
        from apps.users.services.referrals import get_per_active_bonus_decimal

        v = get_per_active_bonus_decimal()
        if not isinstance(v, Decimal):
            v = Decimal(str(v))
        # اگر v بزرگتر از 1 باشه، یعنی به‌صورت bp فرستاده شده، باید تقسیم بر 10000 بشه
        if v >= 1:
            v = v / Decimal(10000)
        return v if v >= 0 else DEFAULT_PER_ACTIVE_BONUS_FRACTION
    except Exception:
        return DEFAULT_PER_ACTIVE_BONUS_FRACTION


class ReferralListView(APIView):
    """
    GET /api/users/referral/list
    لیست ریفرال‌ها با آواتار، نام، وضعیت فعال، تاریخ دعوت و درصد بونس.
    """

    permission_classes = [IsAuthenticated]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "referral"

    def get(self, request):
        rels = (
            ReferralRelation.objects.filter(inviter=request.user)
            .select_related("invitee", "invitee__profile")
            .order_by("-created_at")
        )

        per_active_fraction = _safe_per_active_bonus_decimal()  # 0.0001 یعنی 0.01%
        per_active_percent = per_active_fraction * Decimal(100)

        items = []
        for rel in rels:
            inv = getattr(rel, "invitee", None)
            if not inv:
                items.append(
                    {
                        "user_id": None,
                        "display_name": "Deleted user",
                        "username": "",
                        "avatar_url": "",
                        "invited_at": rel.created_at.isoformat().replace("+00:00", "Z"),
                        "is_active": False,
                        "bonus_rate_percent": 0.0,
                        "bonus_rate_text": "+0.00%",
                    }
                )
                continue

            # نام و یوزرنیم
            prof = getattr(inv, "profile", None)
            display_name = (
                getattr(prof, "display_name", None)
                or getattr(inv, "username", None)
                or getattr(inv, "email", "")
                or f"user-{inv.id}"
            )
            username = getattr(inv, "username", "") or ""
            avatar_url = _avatar_url_from_profile(request, inv)

            # Active status
            is_active = bool(getattr(rel, "active", False))
            row_percent = per_active_percent if is_active else Decimal("0")

            items.append(
                {
                    "user_id": inv.id,
                    "display_name": display_name,
                    "username": username,
                    "avatar_url": avatar_url,
                    "invited_at": rel.created_at.isoformat().replace("+00:00", "Z"),
                    "is_active": is_active,
                    "bonus_rate_percent": float(row_percent),
                    "bonus_rate_text": f"+{row_percent:.2f}%",
                }
            )

        payload = {
            "total_referrals": rels.count(),
            "active_referrals": rels.filter(active=True).count(),
            "per_active_bonus_percent": float(per_active_percent),
            "items": items,
        }
        return Response(payload, status=200)


class ReferralResolveView(APIView):
    """
    GET /api/users/referral/resolve?code=XXXX
    برای لندینگ عمومی: اطلاعات خلاصه دعوت‌کننده را برمی‌گرداند تا UI نشان بدهد.
    """

    authentication_classes = []
    permission_classes = [AllowAny]

    @swagger_auto_schema(
        operation_summary="Resolve referral code (public)",
        manual_parameters=[
            openapi.Parameter(
                "code",
                openapi.IN_QUERY,
                description="Referral code",
                type=openapi.TYPE_STRING,
                required=True,
            )
        ],
        responses={200: ReferralResolveResponseSerializer},
    )
    def get(self, request):
        code = (request.query_params.get("code") or "").strip()
        if not code:
            return Response({"detail": "code is required"}, status=400)

        inv = ReferralInvite.objects.filter(code=code).select_related("inviter").first()
        if not inv:
            payload = {
                "code": code,
                "inviter_id": None,
                "inviter_username": None,
                "inviter_display_name": None,
                "inviter_avatar": None,
                "valid": False,
            }
            return Response(payload, status=200)

        inviter = inv.inviter
        prof = getattr(inviter, "profile", None)
        payload = {
            "code": code,
            "inviter_id": inviter.id if inviter else None,
            "inviter_username": getattr(inviter, "username", None),
            "inviter_display_name": getattr(prof, "display_name", None) if prof else None,
            "inviter_avatar": getattr(prof, "avatar", None) if prof else None,
            "valid": True,
        }
        return Response(payload, status=200)


class ReferralMyCodeView(APIView):
    """
    GET /api/users/referral/my
    اگر کاربر کد نداشته باشد، یکی می‌سازد؛ در غیر اینصورت آخرین کد را برمی‌گرداند.
    """

    permission_classes = [IsAuthenticated]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "auth"

    @swagger_auto_schema(
        operation_summary="Get my referral code (create if none)",
        responses={200: ReferralMyCodeSerializer},
    )
    def get(self, request):
        inv = ReferralInvite.objects.filter(inviter=request.user).order_by("-id").first()
        if not inv:
            inv = ReferralInvite.objects.create(inviter=request.user, code=_gen_ref_code())

        # اگر دامنه لندینگ داری، اینجا بذار
        landing_base = getattr(request, "build_absolute_uri", lambda x: x)("/")
        url = f"{landing_base}?invite={inv.code}"

        data = {
            "code": inv.code,
            "url": url,
            "created_at": inv.created_at,
        }
        ser = ReferralMyCodeSerializer(data)
        return Response(ser.data, status=200)


class ReferralRotateCodeView(APIView):
    """
    POST /api/users/referral/rotate
    کد جدید می‌سازد (کد قبلی همچنان معتبر می‌ماند مگر فیلدی برای غیرفعالسازی داشته باشید).
    """

    permission_classes = [IsAuthenticated]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "auth"

    @swagger_auto_schema(operation_summary="Rotate my referral code (issue a new one)")
    def post(self, request):
        code = _gen_ref_code()
        inv = ReferralInvite.objects.create(inviter=request.user, code=code)

        landing_base = getattr(request, "build_absolute_uri", lambda x: x)("/")
        url = f"{landing_base}?invite={inv.code}"

        data = {
            "code": inv.code,
            "url": url,
            "created_at": inv.created_at,
        }
        ser = ReferralMyCodeSerializer(data)
        return Response(ser.data, status=201)


class EmailChangeRequestView(APIView):
    """
    POST /api/users/email/change/request
    ایمیل جدید می‌گیرد، اگر آزاد بود OTP به آن ارسال می‌کند.
    """

    permission_classes = [IsAuthenticated]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "auth"

    @swagger_auto_schema(
        operation_summary="Request email change (send OTP to new_email)",
        request_body=EmailChangeRequestSerializer,
        responses={200: "ok", 400: "bad request"},
    )
    def post(self, request):
        ser = EmailChangeRequestSerializer(data=request.data)
        ser.is_valid(raise_exception=True)
        new_email = ser.validated_data["new_email"].lower().strip()

        U = get_user_model()
        if U.objects.filter(email__iexact=new_email).exists():
            return Response({"detail": "Email is already in use."}, status=400)

        # OTP بساز/ارسال کن (از util خودت استفاده می‌کنیم)
        # اگر create_otp_for_email امضای purpose می‌پذیرد، بده: purpose="change_email"
        try:
            create_otp_for_email(
                new_email
            )  # یا create_otp_for_email(new_email, purpose="change_email")
        except Exception:
            # در محیط dev اگر ایمیل‌زن نداری، خطا نده
            pass

        # یک رکورد EmailVerification برای ردیابی هم معمولاً ایجاد/به‌روز می‌شود؛
        # فرض می‌کنیم util همین کار را انجام می‌دهد. اگر نه، اینجا می‌توانی ثبت کنی.

        return Response({"detail": "OTP sent to new_email"}, status=200)


class EmailChangeConfirmView(APIView):
    """
    POST /api/users/email/change/confirm
    با new_email + code → اگر معتبر بود ایمیل اکانت را آپدیت می‌کند.
    """

    permission_classes = [IsAuthenticated]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "auth"

    @swagger_auto_schema(
        operation_summary="Confirm email change (verify OTP and update account email)",
        request_body=EmailChangeConfirmSerializer,
        responses={200: "ok", 400: "invalid code"},
    )
    def post(self, request):
        ser = EmailChangeConfirmSerializer(data=request.data)
        ser.is_valid(raise_exception=True)
        new_email = ser.validated_data["new_email"].lower().strip()
        code = ser.validated_data["code"].strip()

        U = get_user_model()
        if U.objects.filter(email__iexact=new_email).exists():
            return Response({"detail": "Email is already in use."}, status=400)

        # اعتبارسنجی OTP
        ok = False
        try:
            # اگر verify_otp امضای (email, code, purpose) می‌خواهد، purpose="change_email" بده
            ok = verify_otp(
                new_email, code
            )  # یا verify_otp(new_email, code, purpose="change_email")
        except Exception:
            ok = False

        if not ok:
            return Response({"detail": "Invalid or expired code."}, status=400)

        # آپدیت ایمیل کاربر
        u = request.user
        u.email = new_email
        u.save(update_fields=["email"])

        return Response({"detail": "email_changed", "email": u.email}, status=200)
