# apps/users/tests/test_referral_flow.py
from decimal import Decimal

from django.contrib.auth import get_user_model
from django.test import TestCase

from apps.miners.models import Miner
from apps.plans.models import Plan
from apps.stakes.models import Stake
from apps.token_app.models import Token
from apps.users.models import ReferralInvite, ReferralRelation
from apps.users.services.referrals import (
    compute_final_percent,
    count_active_referrals,
    recompute_relation_active,
)


class ReferralBonusFlowTest(TestCase):
    def setUp(self):
        User = get_user_model()
        self.inviter = User.objects.create_user(email="inviter@test.local", username="inviter")
        self.invitee = User.objects.create_user(email="invitee@test.local", username="invitee")

        # Token تستی با قیمت دلاری
        self.token = Token.objects.create(symbol="MGC", usd_price=Decimal("1.00"), decimals=18)

        # Plan با فیلدهای اجباری (token دیگر FK نیست)
        self.plan = Plan.objects.create(
            name="Plan Test",
            level=1,
            price=Decimal("100"),
            power=1,
            monthly_reward_percent=Decimal("4.5"),
        )
        # اگر لازم داری Plan هم به Token‌ها وصل باشد:
        if hasattr(self.plan, "tokens"):
            self.plan.tokens.add(self.token)

        # Miner جدید (الان Miner، FK به plan و M2M به tokens دارد)
        self.miner = Miner.objects.create(
            plan=self.plan,
            name="Miner Test",
            power=1,
            is_online=True,
        )
        # لینک M2M به Token
        if hasattr(self.miner, "tokens"):
            self.miner.tokens.add(self.token)

        # یک دعوت‌نامه تستی و Relation
        self.invite = ReferralInvite.objects.create(
            inviter=self.inviter, code="TESTCODE", is_active=True
        )
        self.rel = ReferralRelation.objects.create(
            inviter=self.inviter, invitee=self.invitee, invite=self.invite, active=False
        )

    def test_referral_bonus_changes_with_active_stake(self):
        # در ابتدا باید 0 باشد (invitee هنوز active نیست)
        self.assertEqual(count_active_referrals(self.inviter), 0)
        eff0 = compute_final_percent(self.inviter, Decimal("4.5"))
        # فقط صحت خروجی: باید Decimal برگردد
        self.assertIsInstance(eff0, Decimal)

        # ایجاد یک Stake فعال برای invitee
        Stake.objects.create(
            user=self.invitee,
            miner=self.miner,
            token=self.token,
            amount=Decimal("1000"),
            is_active=True,
        )

        # اکتیو شدن relation را محاسبه/آپدیت کن
        recompute_relation_active(self.invitee)
        self.rel.refresh_from_db()
        self.assertTrue(self.rel.active)

        # الان باید active_referrals = 1 شود
        self.assertEqual(count_active_referrals(self.inviter), 1)

        eff1 = compute_final_percent(self.inviter, Decimal("4.5"))
        self.assertIsInstance(eff1, Decimal)
        # انتظار داریم eff1 >= eff0 باشد چون بونس اضافه شده
        self.assertGreaterEqual(eff1, eff0)

        # حالا Invitee را غیرفعال کن
        Stake.objects.filter(user=self.invitee).update(is_active=False)
        recompute_relation_active(self.invitee)
        self.rel.refresh_from_db()
        self.assertFalse(self.rel.active)
        self.assertEqual(count_active_referrals(self.inviter), 0)

        eff2 = compute_final_percent(self.inviter, Decimal("4.5"))
        self.assertIsInstance(eff2, Decimal)
        # بعد از غیرفعال شدن باید به سطح قبل برگردیم یا کمتر از eff1 باشد
        self.assertLessEqual(eff2, eff1)
