from __future__ import annotations

from decimal import Decimal

from django.core.cache import cache
from django.utils import timezone


from apps.events.models import UserEnergyBoost
from apps.users.models import ReferralRelation
from apps.core.utils.percentages import user_badge_bonus_decimal
from apps.stakes.models import Stake

try:
    from apps.config.models import Config
except Exception:
    Config = None  # fallback

"""
Referral service utilities.

This module centralizes referral business logic:
- computing effective monthly reward percent by stacking referral-based bonuses
- keeping ReferralRelation.active in sync with invitees' active stakes
- (optionally) reading dynamic config from Config with short cache

Public functions:
- compute_final_percent(user, base_percent) -> Decimal
- count_active_referrals(user) -> int
- set_referral_active_flag(invitee_user, active=None) -> bool
- recompute_relation_active(invitee) -> None
"""

# ---- Config keys (basis points = bp) ----
CFG_PER_ACTIVE_BONUS_BP = "REFERRAL_PER_ACTIVE_BONUS_BP"  # مثلا 1 = 0.01%
CFG_BONUS_CAP_BP = "REFERRAL_BONUS_CAP_BP"  # مثلا 150 = 1.50%

# ---- Defaults if Config missing ----
DEFAULT_PER_ACTIVE_BONUS_BP = 1  # 0.01%
DEFAULT_BONUS_CAP_BP = 150  # 1.50%
CFG_BADGE_BONUS_CAP_BP = "BADGE_BONUS_CAP_BP"  # سقف badge bp
CFG_GLOBAL_BONUS_CAP_BP = "GLOBAL_BONUS_CAP_BP"  # سقف مجموع (referral+badge) bp

DEFAULT_BADGE_CAP_BP = 1
DEFAULT_GLOBAL_CAP_BP = 300  # 3.00%  (کل سقف)


def _get_config_int(key: str, default: int) -> int:
    """Read integer from Config with short cache."""
    cache_key = f"cfg_int:{key}"
    cached = cache.get(cache_key)
    if cached is not None:
        return cached
    val = default
    try:
        if Config is not None:
            obj = Config.objects.filter(key=key).first()
            if obj and obj.value is not None:
                val = int(str(obj.value))
    except Exception:
        val = default
    cache.set(cache_key, val, 30)
    return val


# --- Backward-compat shims (keep old imports working) ---


def get_per_active_bonus_decimal():
    """Return per-active referral bonus as Decimal (e.g., 0.0001 for 0.01%)."""
    bp = _get_config_int(CFG_PER_ACTIVE_BONUS_BP, DEFAULT_PER_ACTIVE_BONUS_BP)
    return _bp_to_decimal(bp)


def get_bonus_cap_decimal():
    """Return cap as Decimal (e.g., 0.0150 for 1.50%)."""
    bp = _get_config_int(CFG_BONUS_CAP_BP, DEFAULT_BONUS_CAP_BP)
    return _bp_to_decimal(bp)


def _bp_to_decimal(bp: int) -> Decimal:
    # 100 bp = 1%  →  bp / 10000
    return (Decimal(bp) / Decimal(10000)).quantize(Decimal("0.0001"))


def _get_user_referral_profile_bonus(user) -> Decimal:
    """
    Bonus stored on user.profile.referral_bonus_percent (e.g. 0.015 for 1.5%).
    If missing → 0.
    """
    profile = getattr(user, "profile", None)
    raw_bonus = getattr(profile, "referral_bonus_percent", 0.0) or 0.0
    try:
        return Decimal(str(raw_bonus))
    except Exception:
        return Decimal("0")


def count_active_referrals(user) -> int:
    """
    Return number of active referrals for the inviter.

    An "active" referral is an invitee who currently has at least one active Stake.
    This is reflected in ReferralRelation.active; if you don't persist it, you can
    derive it on-the-fly from Stake, but persisting is cheaper at read time.
    """

    try:
        return int(ReferralRelation.objects.filter(inviter=user, active=True).count())
    except Exception:
        return 0


# ---------- Core formulas ----------


def _compute_bonus_fraction(user) -> Decimal:
    """
    محاسبه‌ی مجموع بونس به‌صورت «ضریب کسری» (مثلاً 0.015 یعنی 1.5%).
    شامل:
      - profile bonus (از پروفایل کاربر، اگر باشد)
      - per-active-referral bonus (از Config، بر حسب bp) با سقف کلی (از Config)
    """
    if not user:
        return Decimal("0")

    # از Config بخوان
    per_active_bp = _get_config_int(CFG_PER_ACTIVE_BONUS_BP, DEFAULT_PER_ACTIVE_BONUS_BP)
    cap_bp = _get_config_int(CFG_BONUS_CAP_BP, DEFAULT_BONUS_CAP_BP)

    per_active = _bp_to_decimal(per_active_bp)  # e.g. 0.0001
    cap = _bp_to_decimal(cap_bp)  # e.g. 0.0150

    n_active = count_active_referrals(user)

    # مجموع بونس فعال‌ها
    active_bonus = per_active * Decimal(n_active)

    # بونس از پروفایل (اگر set شده باشد)
    profile_bonus = _get_user_referral_profile_bonus(user)

    total = active_bonus + profile_bonus
    if total > cap:
        total = cap
    return total






def compute_final_percent(user, base_percent: Decimal, at=None) -> Decimal:
    """
    Calculates the effective mining percent for a user, including:
      - Referral active bonuses (additive)
      - Turbo / Energy boosts (multiplicative)
    """
    if not user:
        return base_percent

    effective = Decimal(str(base_percent or 0))
    now = at or timezone.now()

    # 1) Referral bonus (additive)
    n_active = count_active_referrals(user)
    per_active_bonus = get_per_active_bonus_decimal()
    cap_bonus = get_bonus_cap_decimal()

    referral_bonus = per_active_bonus * Decimal(n_active)
    if referral_bonus > cap_bonus:
        referral_bonus = cap_bonus

    effective += referral_bonus

    # 2) Energy Turbo boosts (multiplicative on top of effective)
    boosts = (
        UserEnergyBoost.objects
        .filter(user=user, is_active=True, starts_at__lte=now, ends_at__gte=now)
        .select_related("definition")
    )

    factor = Decimal("1")
    for ub in boosts:
        p = ub.definition.percent_boost or Decimal("0")  # e.g. 10 => +10%
        factor *= (Decimal("1") + (Decimal(str(p)) / Decimal("100")))

    effective = effective * factor

    return effective.quantize(Decimal("0.0001"))




def compute_final_rate(user, base_rate: float) -> float:
    base = Decimal(str(base_rate or 0.0))
    total_bonus = _compute_bonus_fraction(user)
    return float(base * (Decimal(1) + total_bonus))


def set_referral_active_flag(invitee_user, active: bool | None = None) -> bool:
    """
    Ensure ReferralRelation.active reflects invitee's current stake activity.

    If 'active' is None, it is recomputed by checking if invitee has any active Stake.
    Returns the effective boolean value written to the relation.
    """

    effective_active = active
    if effective_active is None:
        effective_active = Stake.objects.filter(user=invitee_user, is_active=True).exists()

    ReferralRelation.objects.filter(invitee=invitee_user).update(active=effective_active)
    return effective_active


def recompute_referral_active_for_inviter(inviter_user) -> int:
    """
    تعداد inviteeهای فعال یک inviter را برمی‌گرداند (برای نمایش/کش).
    """
    try:
        return int(ReferralRelation.objects.filter(inviter=inviter_user, active=True).count())
    except Exception:
        return 0


def recompute_relation_active(invitee):
    """
    Recompute and persist ReferralRelation.active for a given invitee
    based on whether they have at least one active Stake.
    """

    has_active = Stake.objects.filter(user=invitee, is_active=True).exists()
    rel = ReferralRelation.objects.filter(invitee=invitee).first()
    if rel and rel.active != has_active:
        rel.active = has_active
        rel.save(update_fields=["active"])
        try:
            from apps.badges.services import check_and_grant_referral_badges

            check_and_grant_referral_badges(rel.inviter)
        except Exception:
            pass


def get_current_referral_params():
    """
    output:
      {
        "per_active_bonus": Decimal,  # example 0.0001 : 0.01%
        "bonus_cap": Decimal,         # example 0.0150 : 1.50%
      }
    """
    per_active_bp = _get_config_int(CFG_PER_ACTIVE_BONUS_BP, DEFAULT_PER_ACTIVE_BONUS_BP)
    cap_bp = _get_config_int(CFG_BONUS_CAP_BP, DEFAULT_BONUS_CAP_BP)
    return {
        "per_active_bonus": _bp_to_decimal(per_active_bp),
        "bonus_cap": _bp_to_decimal(cap_bp),
    }


# Legacy names expected elsewhere:
PER_ACTIVE_REFERRAL_BONUS = get_per_active_bonus_decimal()
REFERRAL_BONUS_CAP = get_bonus_cap_decimal()
