from rest_framework import mixins, permissions, status, viewsets
from rest_framework.decorators import action
from rest_framework.response import Response

from django.contrib.auth import get_user_model

from .models import Ticket, TicketMessage, TicketStatus
from .serializers import TicketCreateSerializer, TicketReplySerializer, TicketSerializer
from .email_utils import (
    send_ticket_reply_email_to_user,
    send_ticket_reply_email_to_staff,
)

User = get_user_model()


class IsStaffOrOwner(permissions.BasePermission):
    def has_object_permission(self, request, view, obj):
        if request.user.is_staff:
            return True
        return getattr(obj, "user_id", None) == request.user.id


class TicketViewSet(
    mixins.ListModelMixin,
    mixins.RetrieveModelMixin,
    mixins.CreateModelMixin,
    viewsets.GenericViewSet,
):
    permission_classes = [permissions.IsAuthenticated]
    queryset = Ticket.objects.select_related("user").all()

    def get_serializer_class(self):
        if self.action == "create":
            return TicketCreateSerializer
        return TicketSerializer

    def get_queryset(self):
        qs = super().get_queryset()
        if not self.request.user.is_staff:
            qs = qs.filter(user=self.request.user)
        return qs

    def perform_create(self, serializer):
        ticket = serializer.save(user=self.request.user, status=TicketStatus.OPEN)
        # پیام اولیه‌ی کاربر (اگر نیاز دارید داخل create ارسال شود)
        body = self.request.data.get("message") or ""
        if body.strip():
            TicketMessage.objects.create(
                ticket=ticket, author=self.request.user, from_staff=False, body=body
            )

    from .email_utils import (
        send_ticket_reply_email_to_user,
        send_ticket_reply_email_to_staff,
    )

    @action(
        detail=True,
        methods=["post"],
        url_path="reply",
        permission_classes=[permissions.IsAuthenticated, IsStaffOrOwner],
    )
    def reply(self, request, pk=None):
        ticket = self.get_object()
        ser = TicketReplySerializer(data=request.data)
        ser.is_valid(raise_exception=True)
        msg_body = ser.validated_data["body"]

        from_staff = request.user.is_staff
        message = TicketMessage.objects.create(
            ticket=ticket,
            author=request.user,
            from_staff=from_staff,
            body=msg_body,
        )

        if from_staff and ticket.status != TicketStatus.CLOSED:
            ticket.status = TicketStatus.ANSWERED
            ticket.save(update_fields=["status", "updated_at"])

        # 👇 اینجا حتماً ایمیل را بفرست:
        if from_staff:
            send_ticket_reply_email_to_user(ticket, message)
        else:
            send_ticket_reply_email_to_staff(ticket, message)

        return Response({"ok": True}, status=status.HTTP_201_CREATED)

    @action(
        detail=True,
        methods=["post"],
        url_path="close",
        permission_classes=[permissions.IsAuthenticated, IsStaffOrOwner],
    )
    def close(self, request, pk=None):
        ticket = self.get_object()
        ticket.status = TicketStatus.CLOSED
        ticket.save(update_fields=["status", "updated_at"])
        return Response({"ok": True})
