from decimal import Decimal

from django.conf import settings
from django.db import models
from django.utils.timezone import now

from apps.users.services.referrals import compute_final_percent


def _get_stake_user(stake):
    """
    سعی می‌کند user را از آبجکت stake بیرون بکشد حتی اگر فیلد مستقیم نباشد.
    اگر پیدا نشد، None برمی‌گرداند.
    """
    if not stake:
        return None
    # رایج‌ترین حالت
    if hasattr(stake, "user") and stake.user:
        return stake.user
    # بعضی پروژه‌ها:
    if hasattr(stake, "owner") and stake.owner:
        return stake.owner
    # اگر Stake به Miner وصل است:
    miner = getattr(stake, "miner", None)
    if miner:
        if hasattr(miner, "user") and miner.user:
            return miner.user
        if hasattr(miner, "owner") and miner.owner:
            return miner.owner
    return None


class Reward(models.Model):
    STATUS_CHOICES = [
        ("paid", "Paid"),
        ("failed", "Failed"),
    ]
    user = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.CASCADE)
    miner = models.ForeignKey("miners.Miner", on_delete=models.CASCADE)
    amount = models.DecimalField(max_digits=30, decimal_places=8)
    created_at = models.DateTimeField(auto_now_add=True)
    status = models.CharField(max_length=10, choices=STATUS_CHOICES, default="paid")

    def __str__(self):
        return f"Reward {self.id} for User {self.user_id} - {self.amount}"


class RewardCycle(models.Model):
    stake = models.ForeignKey("stakes.Stake", on_delete=models.CASCADE)
    due_at = models.DateTimeField()
    unlock_time = models.DateTimeField()
    is_paid = models.BooleanField(default=False)
    amount = models.DecimalField(max_digits=30, decimal_places=8, default=Decimal("0"))
    created_at = models.DateTimeField(auto_now_add=True)
    completed = models.BooleanField(default=False)
    reward_percent = models.DecimalField(max_digits=5, decimal_places=2, default=Decimal("4.5"))

    def calculate_amount(self):
        # مقدار استیک
        stake_amount = self.stake.amount if self.stake and self.stake.amount else Decimal("0")

        # پلن از روی stake.miner.plan (در صورت نبود، پیش‌فرض 4.5%)
        plan = getattr(getattr(self.stake, "miner", None), "plan", None)
        percent_base = getattr(plan, "monthly_reward_percent", None) or Decimal("4.5")

        # یوزر مستقیم از stake.user
        user = getattr(self.stake, "user", None)

        # درصد مؤثر با بونس ریفرال
        percent_effective = compute_final_percent(user, percent_base) if user else percent_base
        self.reward_percent = percent_effective

        # مبلغ = amount * (درصد/100)
        calculated_amount = (stake_amount * percent_effective) / Decimal("100")
        self.amount = calculated_amount.quantize(Decimal("0.00000001"))

        self.save(update_fields=["amount", "reward_percent"])
        return self.amount

    def check_and_complete(self):
        if now() >= self.unlock_time:
            if not self.completed:
                self.completed = True
                self.save(update_fields=["completed"])
                return True
        return False

    def __str__(self):
        return f"RewardCycle {self.id} for Stake {self.stake_id} - Completed: {self.completed}"
