from decimal import Decimal

from django.contrib.auth import get_user_model
from django.db.models import Sum
from django.utils import timezone

from apps.miners.models import Miner
from apps.stakes.models import Stake
from apps.token_app.models import Token
from apps.wallets.models import OutgoingTransaction

User = get_user_model()


def get_live_stats():
    online_miners = Miner.objects.filter(is_online=True).count()
    total_users = User.objects.count()

    token_prices = dict(Token.objects.values_list("id", "usd_price"))
    # staked
    staked_usd = Decimal("0")
    staked_rows = Stake.objects.values("token_id", "token__symbol").annotate(total=Sum("amount"))
    staked_breakdown = []
    for r in staked_rows:
        price = token_prices.get(r["token_id"]) or Decimal("0")
        usd = (r["total"] or 0) * price
        staked_usd += usd
        staked_breakdown.append(
            {
                "symbol": r["token__symbol"],
                "amount": str(r["total"] or 0),
                "usd": str(usd.quantize(Decimal("0.01"))),
            }
        )

    # paid
    paid_usd = Decimal("0")
    paid_rows = (
        OutgoingTransaction.objects.filter(status__in=["sent", "approved"])
        .values("token_id", "token__symbol")
        .annotate(total=Sum("amount"))
    )
    paid_breakdown = []
    for r in paid_rows:
        price = token_prices.get(r["token_id"]) or Decimal("0")
        usd = (r["total"] or 0) * price
        paid_usd += usd
        paid_breakdown.append(
            {
                "symbol": r["token__symbol"],
                "amount": str(r["total"] or 0),
                "usd": str(usd.quantize(Decimal("0.01"))),
            }
        )

    return {
        "live": True,
        "updated_at": timezone.now().isoformat(),
        "online_miners": online_miners,
        "total_users": total_users,
        "total_staked_usd": str(staked_usd.quantize(Decimal("0.01"))),
        "total_paid_usd": str(paid_usd.quantize(Decimal("0.01"))),
        "breakdown": {"staked": staked_breakdown, "paid": paid_breakdown},
        "platform_status": "operational",
        "uptime_percent": 99.9,
    }
