from django.contrib import admin
from django.utils.html import format_html
from django import forms
from django.contrib import admin

from .models import Task
from .constants import TaskConditionCode

from .models import (
    EnergyTurboDefinition,
    MinerAccessDefinition,
    Task,
    UserEnergyBoost,
    UserTaskProgress,
    UserTemporaryMinerAccess,
)
class TaskAdminForm(forms.ModelForm):
    """
    فرم سفارشی برای Task که یه فیلد عددی N (required_count) به ادمین نشون می‌ده
    و پشت صحنه این مقدار رو تو condition_params ذخیره می‌کنه.
    """

    required_count = forms.IntegerField(
        required=False,
        min_value=1,
        label="Required count (N)",
        help_text="For example: 5 means the user must perform this event N times in each interval",
    )

    class Meta:
        model = Task
        fields = "__all__"

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        instance = self.instance
        if instance and instance.pk and instance.condition_params:
            rc = instance.condition_params.get("required_count")
            if rc is not None:
                self.fields["required_count"].initial = rc

        # اگر خواستی فقط برای بعضی condition_codeها فیلد را نشان دهی، اینجا می‌توانی شرط بگذاری
        # مثلاً:
        # if instance and instance.condition_code != TaskConditionCode.TICKET_COUNT:
        #     self.fields["required_count"].widget = forms.HiddenInput()

    def clean(self):
        cleaned = super().clean()
        # اگر لازم بود شرط بگذاری که برای بعضی condition_codeها required باشد، اینجا انجام بده.
        # مثلاً:
        # cond = cleaned.get("condition_code")
        # rc = cleaned.get("required_count")
        # if cond == TaskConditionCode.TICKET_COUNT and not rc:
        #     self.add_error("required_count", "برای این نوع تسک، N الزامی است.")
        return cleaned

    def save(self, commit=True):
        """
        اینجا مقدار required_count را در condition_params ذخیره می‌کنیم.
        """
        obj = super().save(commit=False)

        condition_params = obj.condition_params or {}
        rc = self.cleaned_data.get("required_count")

        if rc is not None:
            condition_params["required_count"] = rc
        else:
            condition_params.pop("required_count", None)

        obj.condition_params = condition_params

        if commit:
            obj.save()
        return obj


@admin.register(EnergyTurboDefinition)
class EnergyTurboDefinitionAdmin(admin.ModelAdmin):
    list_display = ["id", "name", "percent_boost", "duration_seconds", "is_active"]
    list_filter = ["is_active"]
    search_fields = ["name"]


@admin.register(MinerAccessDefinition)
class MinerAccessDefinitionAdmin(admin.ModelAdmin):
    list_display = ["id", "name", "miner", "duration_seconds", "is_active"]
    list_filter = ["is_active", "miner"]
    search_fields = ["name", "miner__name"]



@admin.register(Task)
class TaskAdmin(admin.ModelAdmin):
    form = TaskAdminForm
    list_display = [
        "id",
        "title",
        "kind",
        "reward_type",
        "is_active",
        "image_thumb",
        "required_count_display",
    ]
    list_filter = ["kind", "reward_type", "is_active", "condition_code"]
    search_fields = ["title", "slug"]

    readonly_fields = ["image_preview", "condition_params"]

    fieldsets = (
        (None, {
            "fields": ("slug", "title", "description", "image", "image_preview")
        }),
        ("Window", {
            "fields": ("kind", "window_start", "window_end")
        }),
        ("Condition", {
            "fields": (
                "condition_code",
                "required_count",  # ⬅️ این رو اضافه کن
                "condition_params",  # می‌تونی نگهش داری چون readonly شده
                "is_active",
            )
        }),
        ("Reward", {
            "fields": ("reward_type", "reward_content_type", "reward_object_id")
        }),
    )

    def image_thumb(self, obj):
        if obj.image:
            return format_html('<img src="{}" style="height:40px;" />', obj.image.url)
        return "-"
    image_thumb.short_description = "image"

    def image_preview(self, obj):
        if obj.image:
            return format_html('<img src="{}" style="max-height:200px;" />', obj.image.url)
        return "without image"

    def required_count_display(self, obj):
        params = obj.condition_params or {}
        return params.get("required_count", "-")

    required_count_display.short_description = "N (Required count)"


@admin.register(UserTaskProgress)
class UserTaskProgressAdmin(admin.ModelAdmin):
    list_display = [
        "id",
        "user",
        "task",
        "status",
        "completed_at",
        "rewarded_at",
        "window_start",
        "window_end",
    ]
    list_filter = ["status", "task__kind", "task__condition_code"]
    search_fields = ["user__username", "task__title"]


@admin.register(UserEnergyBoost)
class UserEnergyBoostAdmin(admin.ModelAdmin):
    list_display = ["id", "user", "definition", "miner", "starts_at", "ends_at", "is_active"]
    list_filter = ["is_active", "definition"]


@admin.register(UserTemporaryMinerAccess)
class UserTemporaryMinerAccessAdmin(admin.ModelAdmin):
    list_display = ["id", "user", "definition", "starts_at", "ends_at", "is_active"]
    list_filter = ["is_active", "definition"]
