# apps/claims/services.py
from __future__ import annotations
from apps.miners.models import UserMiner  # برای type hint (مشکل خاصی نیست چون ClaimIntent هم ازش استفاده می‌کند)

import json
import logging
import time
from decimal import Decimal, getcontext
from django.db import transaction
from eth_account import Account
from web3 import Web3
from web3.middleware.proof_of_authority import ExtraDataToPOAMiddleware
from django.conf import settings
from django.db import transaction
from django.db.models import Q, Sum
from django.utils import timezone

from apps.claims.models import ClaimIntent, ClaimTx, ClaimPurpose

from apps.token_app.models import Token
from apps.wallets.models import WalletConnection
from typing import Optional
from decimal import Decimal


from apps.stakes.models import Stake
logger = logging.getLogger(__name__)

# ---------- eth-account encoder selection ----------
try:
    # newer API (commonly provides encode_typed_data)
    from eth_account.messages import encode_typed_data as _EIP712_ENCODE

    _ENCODER_NAME = "typed"
except Exception:
    # older API
    from eth_account.messages import encode_structured_data as _EIP712_ENCODE

    _ENCODER_NAME = "structured"

getcontext().prec = 28
SECONDS_IN_30D = Decimal("2592000")  # 30d

_w3 = Web3(Web3.HTTPProvider(settings.RPC_URL))
_w3.middleware_onion.inject(ExtraDataToPOAMiddleware, layer=0)


def _contract():
    from pathlib import Path

    abi = json.loads(Path(__file__).with_name("abi.json").read_text())
    return _w3.eth.contract(
        address=Web3.to_checksum_address(settings.CLAIM_CONTRACT_ADDRESS), abi=abi
    )


def _get_nonce_from_chain(user_address: str) -> int:
    """tries: nonces(address) or nonceOf(address); fallback 0 on error"""
    try:
        c = _contract()
        ua = Web3.to_checksum_address(user_address)
        if hasattr(c.functions, "nonces"):
            return int(c.functions.nonces(ua).call())
        if hasattr(c.functions, "nonceOf"):
            return int(c.functions.nonceOf(ua).call())
        logger.warning("nonce reader not found on contract (no nonces/nonceOf). fallback=0")
        return 0
    except Exception:
        logger.exception("nonce read failed; fallback=0")
        return 0


def _eip712_domain():
    return {
        "name": "RZMining",  # 👈 مثل اسکریپت مدیر
        "version": "1",
        "chainId": int(settings.CLAIM_CHAIN_ID),  # باید 56 باشد
        "verifyingContract": Web3.to_checksum_address(settings.CLAIM_CONTRACT_ADDRESS),
    }


def _eip712_encode(typed_data: dict):
    """
    Cross-version wrapper for eth-account:

    اول تلاش می‌کنیم:
      - encode_*(primitive=typed_data)
      - encode_*(typed_data)  # full_message

    اگر نسخه‌ی نصب‌شده ۳ آرگومانی باشد،
    باید 'EIP712Domain' را از types حذف کنیم و فقط تایپ اصلی را بدهیم:
      encode_typed_data(domain, {PrimaryType: [...]}, message)

    در نهایت اگر هیچ‌کدام جواب نداد، آخرین تلاش با ۴ آرگومان.
    """
    # 1) primitive=...
    try:
        return _EIP712_ENCODE(primitive=typed_data)
    except TypeError:
        pass
    except Exception:
        pass

    # 2) یک آرگومان (full dict)
    try:
        return _EIP712_ENCODE(typed_data)
    except TypeError:
        pass
    except Exception:
        pass

    # 3) نسخه‌های ۳-آرگومانی: domain, message_types, message
    try:
        domain = typed_data["domain"]
        types_full = typed_data["types"]
        primary = typed_data.get("primaryType") or "Claim"
        # فقط تایپ اصلی را پاس بده، بدون EIP712Domain
        msg_types = {primary: types_full[primary]}
        message = typed_data["message"]
        return _EIP712_ENCODE(domain, msg_types, message)
    except KeyError:
        # اگر کلیدها پیدا نشدند، می‌رویم سراغ مسیر بعدی
        pass
    except TypeError:
        # اگر این امضاء وجود نداشت
        pass
    except Exception:
        # اگر fail شد، راهِ بعدی را امتحان می‌کنیم
        pass

    # 4) برخی فورک‌های قدیمی: ۴ آرگومان
    try:
        domain = typed_data["domain"]
        types_full = typed_data["types"]
        primary = typed_data.get("primaryType") or "Claim"
        message = typed_data["message"]
        return _EIP712_ENCODE(domain, types_full, primary, message)
    except Exception as e:
        raise RuntimeError(f"EIP712 encode failed across variants: {e}")


def _build_typed_data(
        to_addr: str, token_addr: str, amount_scaled: int, nonce: int, deadline: int
) -> dict:
    return {
        "types": {
            "Withdraw": [
                {"name": "to", "type": "address"},
                {"name": "token", "type": "address"},
                {"name": "amount", "type": "uint256"},
                {"name": "nonce", "type": "uint256"},
                {"name": "deadline", "type": "uint256"},
            ],
            "EIP712Domain": [
                {"name": "name", "type": "string"},
                {"name": "version", "type": "string"},
                {"name": "chainId", "type": "uint256"},
                {"name": "verifyingContract", "type": "address"},
            ],
        },
        "domain": _eip712_domain(),
        "primaryType": "Withdraw",
        "message": {
            "to": Web3.to_checksum_address(to_addr),
            "token": Web3.to_checksum_address(token_addr),
            "amount": int(amount_scaled),
            "nonce": int(nonce),
            "deadline": int(deadline),
        },
    }


def _sign_typed_data(typed_data: dict) -> str:
    priv = getattr(settings, "CLAIM_SIGNER_PRIVATE_KEY", None) or getattr(
        settings, "SIGNER_PRIVATE_KEY", None
    )
    if not priv:
        raise RuntimeError("SIGNER PRIVATE KEY missing in settings")

    acct = Account.from_key(priv)
    signable = _eip712_encode(typed_data)
    signed = acct.sign_message(signable)

    sig_hex = signed.signature.hex()
    if not sig_hex.startswith("0x"):
        sig_hex = "0x" + sig_hex  # 👈 مطمئن شو 0x دارد
    return sig_hex


def _get_user_wallet(user) -> str | None:
    return (
        WalletConnection.objects.filter(user=user)
            .order_by("-id")
            .values_list("wallet_address", flat=True)
            .first()
    )


def _human_from_scaled(scaled: Decimal | int, decs: int) -> Decimal:
    scaled = Decimal(str(scaled))
    return (scaled / (Decimal(10) ** decs)).quantize(Decimal("0.00000001"))


def _scale_amount(human: Decimal, decs: int) -> int:
    return int((Decimal(str(human)) * (Decimal(10) ** decs)).to_integral_value())


def _base_percent_from_plan(miner) -> Decimal:
    try:
        plan = getattr(miner, "plan", None)
        percent = getattr(plan, "monthly_reward_percent", None)
        if percent is None:
            return Decimal("1.0")
        return Decimal(str(percent))
    except Exception:
        return Decimal("1.0")


def _safe_compute_final_percent(user, base_percent: Decimal, as_of=None) -> Decimal:
    """
    اگر سرویس رفرال شما compute_final_percent(at=...) را پشتیبانی کند، از زمانِ as_of استفاده می‌کنیم
    تا گذشته با نرخ جدید بازحساب نشود. در غیر این صورت fallback به نسخه فعلی.
    """
    try:
        from apps.users.services.referrals import compute_final_percent
    except Exception:
        return base_percent

    try:
        # نسخه‌ی زمان‌مند (ترجیحی اگر موجود باشد)
        if as_of is not None:
            return Decimal(str(compute_final_percent(user, base_percent, at=as_of)))
        # نسخه‌ی قدیمی بدون زمان
        return Decimal(str(compute_final_percent(user, base_percent)))
    except TypeError:
        # امضای تابع پارامتر زمانی ندارد
        try:
            return Decimal(str(compute_final_percent(user, base_percent)))
        except Exception:
            return base_percent
    except Exception:
        return base_percent



def get_pending_by_symbol_for_user(
    user,
    symbol: str,
    stake: Optional[Stake] = None,
) -> Decimal:
    """
    accrued (براساس استیک‌ها، با احترام به pause و امکان نرخِ زمان‌مند)
    منهای withdrawn (ClaimTx)
    → خروجی: مقدار انسان‌خوان با دقت token.decimals
    """
    token = Token.objects.filter(symbol__iexact=symbol).first()
    if not token:
        return Decimal("0")

    decs = int(getattr(token, "decimals", 18) or 18)
    now = timezone.now()

    # Accrued
    accrued = Decimal("0")

    stakes = (
        Stake.objects.filter(
            user=user,
            token__symbol__iexact=symbol,
            is_active=True,
        )
        .select_related("miner", "token")
        .order_by("id")
    )

    if stake is not None:
        stakes = stakes.filter(pk=stake.pk)

    if not stakes.exists():
        return Decimal("0")

    for st in stakes:
        try:
            amount = Decimal(str(st.amount))
        except Exception:
            amount = Decimal("0")

        started_at = getattr(st, "created_at", None)
        paused_at = getattr(st, "paused_at", None)
        window_end = paused_at or now
        if started_at is None or window_end <= started_at:
            continue

        seconds_active = Decimal(int((window_end - started_at).total_seconds()))
        if seconds_active <= 0:
            continue

        miner = getattr(st, "miner", None)
        base_percent = _base_percent_from_plan(miner) if miner else Decimal("1.0")
        effective_percent = _safe_compute_final_percent(
            user,
            base_percent,
            as_of=window_end,
        )

        rate_fraction = Decimal(str(effective_percent)) / Decimal("100")
        earning_per_second = (amount * rate_fraction) / SECONDS_IN_30D
        accrued += earning_per_second * seconds_active

    # Withdrawn
    user_wallet = (
        WalletConnection.objects.filter(user=user)
        .values_list("wallet_address", flat=True)
        .order_by("-id")
        .first()
    )

    claim_filter = Q(user=user) | Q(user__isnull=True)
    if user_wallet:
        claim_filter |= Q(user_address__iexact=user_wallet)

    tok_addr = (
        getattr(token, "contract_address", None)
        or getattr(token, "token_address", None)
        or ""
    ).lower()

    agg = ClaimTx.objects.filter(
        claim_filter,
        token_address__iexact=tok_addr,
    ).aggregate(total_scaled=Sum("amount_scaled"))

    withdrawn_scaled_sum = Decimal(str(agg.get("total_scaled") or 0))
    withdrawn_human = _human_from_scaled(withdrawn_scaled_sum, decs)

    pending = accrued - withdrawn_human
    return pending if pending > 0 else Decimal("0")


def _scaled_from_human(amount_human: Decimal, decimals: int) -> int:
    return int(
        (Decimal(amount_human) * (Decimal(10) ** Decimal(int(decimals)))).quantize(Decimal("1"))
    )


def _is_signature_binary_field() -> bool:
    field = ClaimIntent._meta.get_field("signature")
    return getattr(field, "get_internal_type", lambda: "")() == "BinaryField"


# --- signature helpers ---
def _signature_to_hex(sig) -> str:
    """
    هر چیزی (bytes/str با یا بی‌ِ 0x) بدهی، خروجی همیشه '0x' + hex lower برمی‌گردد.
    """
    if isinstance(sig, bytes | bytearray):
        h = sig.hex()
    else:
        h = str(sig or "")
        if h.startswith("0x") or h.startswith("0X"):
            h = h[2:]
    # بعضی DB ها امضا را uppercase ذخیره می‌کنند؛ normalize به lower
    return "0x" + h.lower()


def _signature_to_store(sig_hex: str):
    """
    اگر فیلد signature از نوع BinaryField باشد، bytes ذخیره می‌کنیم،
    در غیر این‌صورت همان hex (بدون تغییر).
    """
    try:
        field = ClaimIntent._meta.get_field("signature")
        is_binary = field.get_internal_type() == "BinaryField"
    except Exception:
        is_binary = False

    h = str(sig_hex or "")
    if h.startswith("0x") or h.startswith("0X"):
        h = h[2:]

    return bytes.fromhex(h) if is_binary else ("0x" + h)



@transaction.atomic
def make_claim_signature(
    user,
    symbol: str,
    amount_requested: Optional[Decimal],
    *,
    purpose: ClaimPurpose = ClaimPurpose.WITHDRAW,
    target_miner: Optional[UserMiner] = None,
    stake: Optional[Stake] = None,
) -> dict:
    # 0) wallet
    wallet = _get_user_wallet(user)
    if not wallet:
        raise ValueError("wallet_not_connected")

    # 1) token
    token = Token.objects.filter(symbol__iexact=symbol).first()
    if not token:
        raise ValueError("unknown_token")

    # 2) مبلغ مجاز (pending) - حالا می‌تونه per-stake هم باشه
    pending_human = get_pending_by_symbol_for_user(user, symbol, stake=stake)
    if pending_human <= 0:
        raise ValueError("nothing_to_claim")

    if amount_requested is None:
        amount_human = Decimal(pending_human)
    else:
        if amount_requested <= 0:
            raise ValueError("bad_amount")
        amount_human = min(Decimal(amount_requested), Decimal(pending_human))

    decs = int(getattr(token, "decimals", 18) or 18)
    amount_scaled = _scale_amount(amount_human, decs)

    # 3) nonce & deadline
    nonce = _get_nonce_from_chain(wallet)
    now_ts = int(time.time())
    ttl = int(getattr(settings, "CLAIM_DEADLINE_SECONDS", 300))
    deadline = now_ts + ttl

    # 4) ایدمپوتنسی روی (user, nonce) دقیقا مطابق unique constraint
    intent_qs = ClaimIntent.objects.select_for_update().filter(
        user=user,
        nonce=nonce,
    )
    intent: Optional[ClaimIntent] = intent_qs.order_by("-id").first()

    def _payload_from(intent_obj: ClaimIntent, signature_hex_out: str):
        return {
            "symbol": getattr(token, "symbol", ""),
            "to": wallet,
            "token": Web3.to_checksum_address(
                token.contract_address or token.token_address
            ),
            "amount_scaled": str(intent_obj.amount_scaled),
            "nonce": int(intent_obj.nonce),
            "deadline": int(intent_obj.deadline),
            "signature": signature_hex_out,  # همیشه 0x...
            "chain_id": int(getattr(settings, "CLAIM_CHAIN_ID", 56)),
            "verifying_contract": Web3.to_checksum_address(
                settings.CLAIM_CONTRACT_ADDRESS
            ),
            "intent_id": intent_obj.id,
            "expires_in_seconds": max(0, int(intent_obj.deadline) - now_ts),
        }

    # 4.a) اگر intent موجود و معتبر است → همان را برگردان
    if (
        intent
        and intent.status in ("pending", "issued")
        and int(intent.deadline) > now_ts
    ):
        sig_hex_existing = _signature_to_hex(intent.signature)  # ← همیشه 0x
        return _payload_from(intent, sig_hex_existing)

    # 4.b) امضاء جدید بساز (برای intent جدید یا تمدید intent منقضی‌شده)
    typed = _build_typed_data(
        wallet,
        token.contract_address or token.token_address,
        amount_scaled,
        nonce,
        deadline,
    )

    signature_hex = _sign_typed_data(typed)  # 0x...
    signature_to_store = _signature_to_store(signature_hex)

    if intent:
        # تمدید/بروزرسانی intent قبلی (بدون Duplicate)
        intent.token_address = token.contract_address or token.token_address
        if hasattr(intent, "token_symbol"):
            intent.token_symbol = getattr(token, "symbol", "")
        intent.user_address = wallet
        intent.amount_scaled = amount_scaled
        intent.deadline = deadline
        intent.signature = signature_to_store
        intent.status = "pending"
        intent.purpose = purpose
        intent.target_miner = target_miner
        intent.save(
            update_fields=[
                "token_address",
                *(["token_symbol"] if hasattr(intent, "token_symbol") else []),
                "user_address",
                "amount_scaled",
                "deadline",
                "signature",
                "status",
                "purpose",
                "target_miner",
                "updated_at",
            ]
        )
        return _payload_from(intent, _signature_to_hex(signature_hex))

    # 4.c) ساخت intent جدید
    intent = ClaimIntent.objects.create(
        user=user,
        token_address=(token.contract_address or token.token_address),
        token_symbol=(
            getattr(token, "symbol", "")
            if hasattr(ClaimIntent, "token_symbol")
            else ""
        ),
        user_address=wallet,
        amount_scaled=amount_scaled,
        nonce=nonce,
        deadline=deadline,
        signature=signature_to_store,
        status="pending",
        purpose=purpose,
        target_miner=target_miner,
    )

    return _payload_from(intent, _signature_to_hex(signature_hex))



def upsert_claim_success_from_event(
        *, user_address: str, token_address: str, amount_scaled: int, nonce: int, tx_hash: str
) -> ClaimTx:
    """
    نسخهٔ مینیمال سازگار با مدل فعلی ClaimTx (بدون status/deadline/...).
    اگر رکوردی با همین tx_hash وجود داشته باشد، همان را برمی‌گرداند.
    وگرنه یک رکورد جدید می‌سازد (با مقدار amount_scaled و ...).
    """
    ua = (user_address or "").lower()
    ta = Web3.to_checksum_address(token_address)

    with transaction.atomic():
        # اگر قبلاً با همین tx_hash ثبت شده، همان را بده
        existing = ClaimTx.objects.filter(tx_hash=tx_hash).first()
        if existing:
            return existing

        # نماد را اگر بشود حدس بزنیم
        tok = (
                Token.objects.filter(contract_address__iexact=ta).first()
                or Token.objects.filter(token_address__iexact=ta).first()
        )
        token_symbol = tok.symbol if tok else ""

        # تلاش برای نگاشت user از کیف‌پول (اختیاری)
        user_id = (
            WalletConnection.objects.filter(wallet_address__iexact=ua)
                .values_list("user_id", flat=True)
                .first()
        )

        inst = ClaimTx.objects.create(
            user_id=user_id,
            user_address=ua,
            token_address=ta,
            token_symbol=token_symbol or "UNKNOWN",
            amount_scaled=Decimal(str(amount_scaled)),
            nonce=int(nonce),
            tx_hash=tx_hash or "",
        )
        return inst
