from django.conf import settings
from django.db import models
from django.utils import timezone

User = settings.AUTH_USER_MODEL


class CampaignStatus(models.TextChoices):
    DRAFT = "draft", "Draft"
    SCHEDULED = "scheduled", "Scheduled"
    SENDING = "sending", "Sending"
    COMPLETED = "completed", "Completed"
    FAILED = "failed", "Failed"


class EmailCampaign(models.Model):
    subject = models.CharField(max_length=255)
    text_content = models.TextField(help_text="The plain text of the campaign message")
    created_at = models.DateTimeField(default=timezone.now)
    scheduled_at = models.DateTimeField(null=True, blank=True)
    status = models.CharField(
        max_length=20, choices=CampaignStatus.choices, default=CampaignStatus.DRAFT
    )
    total_recipients = models.PositiveIntegerField(default=0)
    sent_count = models.PositiveIntegerField(default=0)
    error_count = models.PositiveIntegerField(default=0)
    image = models.ImageField(
        upload_to="broadcast/images/",
        null=True,
        blank=True,
        help_text="Optional image to include in the email (attached, not inline).",
    )
    attachment = models.FileField(
        upload_to="broadcast/attachments/",
        null=True,
        blank=True,
        help_text="Optional file (PDF, ZIP, etc.) to attach to the email.",
    )

    def __str__(self):
        return f"{self.subject} ({self.status})"


class EmailDelivery(models.Model):
    campaign = models.ForeignKey(EmailCampaign, on_delete=models.CASCADE, related_name="deliveries")
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    sent = models.BooleanField(default=False)
    error_message = models.TextField(blank=True, null=True)
    sent_at = models.DateTimeField(null=True, blank=True)

    class Meta:
        indexes = [
            models.Index(fields=["campaign", "sent"]),
        ]
