# apps/wallets/services/balances.py
from decimal import Decimal

from web3 import Web3

from django.conf import settings

# ERC20 ABI مینیمال برای balanceOf
_ERC20_ABI = [
    {
        "constant": True,
        "inputs": [{"name": "_owner", "type": "address"}],
        "name": "balanceOf",
        "outputs": [{"name": "balance", "type": "uint256"}],
        "type": "function",
    }
]


def _w3():
    rpc = getattr(settings, "RPC_URL", None) or getattr(
        settings, "BSC_RPC_URL", "https://bsc-dataseed.binance.org/"
    )
    return Web3(Web3.HTTPProvider(rpc))


def _get_user_wallet_address(user):
    # اگر WalletConnection مدل/نام متفاوتی دارد، اصلاح کن
    from apps.wallets.models import WalletConnection

    wc = WalletConnection.objects.filter(user=user).order_by("-id").first()
    return getattr(wc, "wallet_address", None) if wc else None


def _div_decimals(raw_int, decimals: int) -> Decimal:
    if raw_int is None:
        return Decimal(0)
    scale = Decimal(10) ** Decimal(decimals or 0)
    return (Decimal(int(raw_int)) / scale).quantize(Decimal("0.000000000000000001"))


def get_balances_for_user(user, tokens) -> dict:
    """
    خروجی: { "MGC": Decimal(...), "RZ": Decimal(...), ... }
    اگر آدرس کیف‌پول یا آدرس توکن نامعتبر باشد، مقدار صفر می‌دهد اما خطا نمی‌سازد.
    """
    out: dict[str, Decimal] = {}
    wallet = _get_user_wallet_address(user)
    if not wallet:
        # کیف‌پول متصل نیست ⇒ همه صفر
        for t in tokens:
            if t and getattr(t, "symbol", None):
                out[t.symbol] = Decimal(0)
        return out

    w3 = _w3()
    wallet = Web3.to_checksum_address(wallet)

    for t in tokens:
        try:
            symbol = getattr(t, "symbol", None)
            if not symbol:
                continue
            address = getattr(
                t, "contract_address", None
            )  # مطمئن شو Token.address در مدل هست؛ اگر نیست، صفر می‌زنیم
            decimals = int(getattr(t, "decimals", 18) or 18)

            if not address:
                out[symbol] = Decimal(0)
                continue

            token_addr = Web3.to_checksum_address(address)
            contract = w3.eth.contract(address=token_addr, abi=_ERC20_ABI)

            raw = contract.functions.balanceOf(wallet).call()
            out[symbol] = _div_decimals(raw, decimals)
        except Exception:
            # هر خطایی ⇒ صفر. مهم: به هیچ‌وجه ۵۰۰ نده.
            out[symbol] = Decimal(0)

    return out
