# apps/users/serializers.py

from rest_framework import serializers

from django.contrib.auth import get_user_model

from apps.users.constants import DEFAULT_AVATAR_FALLBACK, DEFAULT_AVATAR_KEYS
from apps.users.models import ReferralInvite

AVATAR_CHOICES = [(k, k) for k in DEFAULT_AVATAR_KEYS]

User = get_user_model()


class EmailRequestSerializer(serializers.Serializer):
    email = serializers.EmailField()


class VerifyOTPRequestSerializer(serializers.Serializer):
    email = serializers.EmailField()
    otp_code = serializers.CharField()


class UsernameUpdateSerializer(serializers.Serializer):
    email = serializers.EmailField()
    username = serializers.CharField(min_length=3)
    password = serializers.CharField(required=False, allow_blank=True, min_length=6)
    referred_by = serializers.CharField(required=False, allow_null=True, allow_blank=True)


class LoginSerializer(serializers.Serializer):
    email = serializers.EmailField()
    password = serializers.CharField()


class ResetPasswordSerializer(serializers.Serializer):
    token = serializers.CharField(max_length=16)
    password = serializers.CharField(min_length=6)


class GoogleAuthSerializer(serializers.Serializer):
    access_token = serializers.CharField()


class TokenSerializer(serializers.Serializer):
    access_token = serializers.CharField()
    token_type = serializers.CharField(default="bearer")


class UserResponseSerializer(serializers.ModelSerializer):
    is_admin = serializers.SerializerMethodField()

    class Meta:
        model = User
        fields = [
            "id",
            "email",
            "username",
            "is_active",
            "is_verified",
            "address",
            "is_admin",  # ← این باید اضافه بشه
        ]
        read_only_fields = ["email"]

    def get_is_admin(self, obj):
        return obj.is_staff


class ReferralCodeSerializer(serializers.Serializer):
    referral_code = serializers.CharField()


class ReferralUserSerializer(serializers.ModelSerializer):
    class Meta:
        model = User
        fields = ["email", "username"]


class WalletSignupSerializer(serializers.Serializer):
    # امضای EVM + nonce اجباری (تأیید امضا در View انجام می‌شود)
    address = serializers.CharField()
    signature = serializers.CharField()
    nonce_id = serializers.IntegerField(required=True)

    provider = serializers.ChoiceField(choices=["metamask", "walletconnect"], default="metamask")

    # الزامی طبق خواسته‌ی مدیر
    email = serializers.EmailField(required=True)
    username = serializers.CharField(required=True, min_length=3, max_length=30)
    avatar_key = serializers.ChoiceField(
        required=False, choices=AVATAR_CHOICES, allow_blank=True, allow_null=True
    )

    # لینک دعوت (اختیاری)
    invite = serializers.CharField(required=False, allow_blank=True)

    # --- normalizers ---
    def validate_username(self, v: str) -> str:
        v = (v or "").strip()
        if not v:
            raise serializers.ValidationError("username is required.")
        if User.objects.filter(username__iexact=v).exists():
            raise serializers.ValidationError("username already taken.")
        return v

    def validate_email(self, v: str) -> str:
        v = (v or "").strip().lower()
        if User.objects.filter(email__iexact=v).exists():
            raise serializers.ValidationError("email already in use.")
        return v

    def validate_address(self, v: str) -> str:
        v = (v or "").strip().lower()
        if not v:
            raise serializers.ValidationError("address is required.")
        # (اختیاری) حداقل چک فرمت: 0x و طول 42
        if not (v.startswith("0x") and len(v) == 42):
            # اگر نمی‌خوای سخت‌گیری این‌جا باشه، این بخش رو حذف کن.
            raise serializers.ValidationError("invalid EVM address format.")
        return v

    def validate_provider(self, v: str) -> str:
        return (v or "metamask").strip().lower()

    def validate_avatar_key(self, v: str) -> str:
        if not v:
            return DEFAULT_AVATAR_FALLBACK
        if v not in DEFAULT_AVATAR_KEYS:
            raise serializers.ValidationError("invalid avatar_key")
        return v

    def validate(self, attrs):
        invite = (attrs.get("invite") or "").strip()
        if invite:
            from apps.users.models import ReferralInvite

            inv = ReferralInvite.objects.filter(code=invite, is_active=True).first()
            if not inv:
                raise serializers.ValidationError({"invite": "invalid_or_inactive"})
            # اگر سقف مصرف دارد، چک کن
            if inv.max_uses is not None and inv.used_count is not None:
                if inv.used_count >= inv.max_uses:
                    raise serializers.ValidationError({"invite": "max_uses_reached"})
            # پاس بده به view
            attrs["_invite_obj"] = inv
        return attrs

    def create(self, validated_data):
        # ساخت واقعی user در View انجام می‌شود (پس فقط دیتا را بازمی‌گردانیم)
        return validated_data


class ProfileUpdateSerializer(serializers.Serializer):
    email = serializers.EmailField(required=False)
    username = serializers.CharField(required=False, min_length=3, max_length=30)
    display_name = serializers.CharField(required=False, max_length=50)
    avatar_key = serializers.ChoiceField(
        required=False, choices=AVATAR_CHOICES, allow_blank=True, allow_null=True
    )

    def validate_email(self, v):
        v = (v or "").strip().lower()
        if not v:
            return v
        # ایمیل تکراری نباشه (به جز خود کاربر)
        me = self.context.get("request").user
        if User.objects.filter(email__iexact=v).exclude(id=me.id).exists():
            raise serializers.ValidationError("email already in use.")
        return v

    def validate_username(self, v):
        v = (v or "").strip()
        if not v:
            return v
        me = self.context.get("request").user
        if User.objects.filter(username__iexact=v).exclude(id=me.id).exists():
            raise serializers.ValidationError("username already taken.")
        return v

    def validate_avatar_key(self, v: str) -> str:
        if not v:
            return DEFAULT_AVATAR_FALLBACK
        if v not in DEFAULT_AVATAR_KEYS:
            raise serializers.ValidationError("invalid avatar_key")
        return v


class ReferralInviteCreateSerializer(serializers.Serializer):
    max_uses = serializers.IntegerField(required=False, min_value=1)
    ttl_hours = serializers.IntegerField(required=False, min_value=1)


class ReferralInviteSerializer(serializers.Serializer):
    code = serializers.CharField()
    url = serializers.CharField()
    is_active = serializers.BooleanField()
    max_uses = serializers.IntegerField(allow_null=True)
    used_count = serializers.IntegerField()
    expires_at = serializers.DateTimeField(allow_null=True)

    class Meta:
        model = ReferralInvite
        fields = ["code", "max_uses", "use_count", "is_active", "expires_at", "created_at"]


class WalletLoginSerializer(serializers.Serializer):
    address = serializers.CharField()
    signature = serializers.CharField()
    nonce_id = serializers.IntegerField()
    provider = serializers.ChoiceField(choices=["metamask", "walletconnect"], default="metamask")

    def validate_address(self, addr):
        addr = (addr or "").strip()
        if not addr:
            raise serializers.ValidationError("address is required.")
        return addr


class ReferralInviteResponseSerializer(serializers.Serializer):
    code = serializers.CharField()
    link = serializers.CharField()


class ReferralRotateResponseSerializer(serializers.Serializer):
    old_code = serializers.CharField()
    new_code = serializers.CharField()
    link = serializers.CharField()


class ReferralMyCodeSerializer(serializers.Serializer):
    code = serializers.CharField()
    url = serializers.CharField(allow_null=True)
    created_at = serializers.DateTimeField()


class ReferralStatsSerializer(serializers.Serializer):
    total_referrals = serializers.IntegerField()
    active_referrals = serializers.IntegerField()
    last_invited_at = serializers.DateTimeField(allow_null=True)


class ReferralListItemSerializer(serializers.Serializer):
    user_id = serializers.IntegerField()
    display_name = serializers.CharField(allow_null=True)
    username = serializers.CharField(allow_null=True)
    avatar_url = serializers.CharField(allow_blank=True)
    invited_at = serializers.DateTimeField()
    is_active = serializers.BooleanField()
    bonus_rate_percent = serializers.FloatField()  # مثلا 0.01 (معادل +0.01%)
    bonus_rate_text = serializers.CharField()  # مثلا "+0.01%"


class ReferralListResponseSerializer(serializers.Serializer):
    total_referrals = serializers.IntegerField()
    active_referrals = serializers.IntegerField()
    per_active_bonus_percent = serializers.FloatField()  # مثلا 0.01 (%)
    items = ReferralListItemSerializer(many=True)


class ReferralResolveResponseSerializer(serializers.Serializer):
    code = serializers.CharField()
    inviter_id = serializers.IntegerField(allow_null=True)
    inviter_username = serializers.CharField(allow_null=True)
    inviter_display_name = serializers.CharField(allow_null=True)
    inviter_avatar = serializers.CharField(allow_null=True)
    valid = serializers.BooleanField()


class EmailChangeRequestSerializer(serializers.Serializer):
    new_email = serializers.EmailField()


class EmailChangeConfirmSerializer(serializers.Serializer):
    new_email = serializers.EmailField()
    code = serializers.CharField(min_length=4, max_length=10)
