# apps/users/models.py
# ── Standard Library
import secrets
import string
from datetime import timedelta

# ── Third-Party
from rest_framework_simplejwt.tokens import RefreshToken

from django.conf import settings
from django.contrib.auth.models import AbstractBaseUser, BaseUserManager, PermissionsMixin

# ── Django
from django.db import models
from django.utils import timezone

from apps.users.constants import DEFAULT_AVATAR_FALLBACK, DEFAULT_AVATAR_KEYS

# ── Project


class UserManager(BaseUserManager):
    def create_user(self, email, username=None, password=None, **extra_fields):
        if not email:
            raise ValueError("The Email field must be set")
        email = self.normalize_email(email)
        user = self.model(email=email, username=username, **extra_fields)
        if password:
            user.set_password(password)
        else:
            user.set_unusable_password()
        user.save(using=self._db)
        return user

    def create_superuser(self, email, username, password=None, **extra_fields):
        extra_fields.setdefault("is_staff", True)
        extra_fields.setdefault("is_superuser", True)
        extra_fields.setdefault("is_active", True)

        return self.create_user(email, username, password, **extra_fields)


class User(AbstractBaseUser, PermissionsMixin):
    email = models.EmailField(unique=True)
    username = models.CharField(max_length=150, unique=True, null=False, blank=False)
    is_active = models.BooleanField(default=True)
    is_staff = models.BooleanField(default=False)
    is_verified = models.BooleanField(default=False)

    address = models.TextField(null=True, blank=True)

    referral_code = models.CharField(max_length=32, unique=True, null=True, blank=True)
    referred_by = models.CharField(max_length=32, null=True, blank=True)
    referral_bonus = models.DecimalField(max_digits=5, decimal_places=2, default=0)  # فیلد جدید

    date_joined = models.DateTimeField(auto_now_add=True)

    objects = UserManager()

    USERNAME_FIELD = "email"
    REQUIRED_FIELDS = []

    def __str__(self):
        return self.email

    def get_tokens(self):
        refresh = RefreshToken.for_user(self)
        return {
            "refresh": str(refresh),
            "access": str(refresh.access_token),
        }


def default_expiry():
    return timezone.now() + timedelta(minutes=5)


class EmailVerification(models.Model):
    email = models.EmailField()
    otp_code = models.CharField(max_length=6)
    is_used = models.BooleanField(default=False)
    expires_at = models.DateTimeField()
    otp_type = models.CharField(max_length=50, default="general")  # Add default here
    created_at = models.DateTimeField(default=timezone.now)  # اینجا default بذار

    def save(self, *args, **kwargs):
        if not self.expires_at:
            self.expires_at = timezone.now() + timedelta(minutes=5)
        super().save(*args, **kwargs)


class ReferralBonusConfig(models.Model):
    bonus_percentage = models.DecimalField(
        max_digits=5, decimal_places=2, default=1
    )  # درصد سود برای دعوت
    updated_at = models.DateTimeField(auto_now=True)

    def __str__(self):
        return f"Referral Bonus: {self.bonus_percentage}%"


class Referral(models.Model):
    inviter = models.ForeignKey(User, related_name="invites", on_delete=models.CASCADE)
    invitee = models.OneToOneField(User, related_name="invited_by", on_delete=models.CASCADE)
    created_at = models.DateTimeField(auto_now_add=True)

    # فعال: وقتی invitee واقعاً شروع به ماین/استفاده می‌کند
    activated_at = models.DateTimeField(null=True, blank=True)
    is_active = models.BooleanField(default=False)

    class Meta:
        db_table = "users_referral"
        indexes = [
            models.Index(fields=["inviter", "is_active"]),
        ]

    def mark_active(self):
        if not self.is_active:
            self.is_active = True
            self.activated_at = timezone.now()
            self.save(update_fields=["is_active", "activated_at"])
            # افزایش پاداش دعوت‌کننده با رعایت سقف:
            increment_referral_bonus(self.inviter)


def increment_referral_bonus(user: User):
    """
    با هر ریفرال فعال، درصد پاداش کاربر افزایش می‌یابد تا سقف.
    اگر ریفرال بعداً غیر فعال شد، درصد فعلی کاهش پیدا نمی‌کند (طبق اکسل).
    """

    step = settings.REFERRAL_BONUS_STEP
    cap = settings.REFERRAL_BONUS_CAP

    profile, _ = UserProfile.objects.get_or_create(
        user=user
    )  # اگر پروفایل داری، از همان استفاده کن
    new_bonus = min((profile.referral_bonus_percent or 0.0) + step, cap)
    profile.referral_bonus_percent = new_bonus
    profile.save(update_fields=["referral_bonus_percent"])


class UserProfile(models.Model):
    user = models.OneToOneField(User, on_delete=models.CASCADE, related_name="profile")
    referral_bonus_percent = models.FloatField(default=0)

    display_name = models.CharField(max_length=50, blank=True, null=True)

    avatar_key = models.CharField(
        max_length=50,
        choices=[(k, k) for k in DEFAULT_AVATAR_KEYS],
        default=DEFAULT_AVATAR_FALLBACK,  # دقت کن: اینجا کلید نه URL
    )

    def __str__(self):
        return f"Profile({self.user_id})"


def generate_invite_code(length=10):
    alphabet = string.ascii_letters + string.digits
    return "".join(secrets.choice(alphabet) for _ in range(length))


class ReferralInvite(models.Model):
    inviter = models.ForeignKey(
        settings.AUTH_USER_MODEL, on_delete=models.CASCADE, related_name="referral_invites"
    )
    code = models.CharField(max_length=32, unique=True, db_index=True)  # unique invite code
    is_active = models.BooleanField(default=True)
    max_uses = models.PositiveIntegerField(null=True, blank=True)  # None => unlimited
    used_count = models.PositiveIntegerField(default=0)
    expires_at = models.DateTimeField(null=True, blank=True)
    created_at = models.DateTimeField(auto_now_add=True)

    @staticmethod
    def generate_code() -> str:
        return secrets.token_urlsafe(12)[:16]  # کوتاه و امن

    @classmethod
    def create_for(cls, inviter, *, max_uses=None, ttl_hours=None):
        code = cls.generate_code()
        expires = None
        if ttl_hours:
            expires = timezone.now() + timezone.timedelta(hours=ttl_hours)
        return cls.objects.create(
            inviter=inviter,
            code=code,
            is_active=True,
            max_uses=max_uses,
            expires_at=expires,
        )

    def can_be_used(self) -> bool:
        if not self.is_active:
            return False
        if self.expires_at and timezone.now() > self.expires_at:
            return False
        if self.max_uses is not None and self.used_count >= self.max_uses:
            return False
        return True

    def mark_used(self):
        self.used_count = (self.used_count or 0) + 1
        if self.max_uses is not None and self.used_count >= self.max_uses:
            self.is_active = False
        self.save(update_fields=["used_count", "is_active"])


class ReferralRelation(models.Model):
    inviter = models.ForeignKey(
        settings.AUTH_USER_MODEL, on_delete=models.CASCADE, related_name="referrals_made"
    )
    invitee = models.OneToOneField(
        settings.AUTH_USER_MODEL, on_delete=models.CASCADE, related_name="referral_taken"
    )
    invite = models.ForeignKey(
        ReferralInvite, on_delete=models.SET_NULL, null=True, blank=True, related_name="relations"
    )
    active = models.BooleanField(
        default=True
    )  # اگر بعداً invitee دیگر فعال نبود/ماین نمی‌کند، false می‌کنیم
    created_at = models.DateTimeField(auto_now_add=True)
    deactivated_at = models.DateTimeField(null=True, blank=True)

    class Meta:
        constraints = [
            models.UniqueConstraint(fields=["invitee"], name="uniq_referral_per_invitee"),
        ]
