# ── Django
from django.contrib import admin
from django.utils.safestring import mark_safe
from django.contrib.auth import get_user_model
from django.contrib import messages

# ── Project
from .constants import DEFAULT_AVATAR_FALLBACK, DEFAULT_AVATARS
from .models import EmailVerification, ReferralBonusConfig, User, UserProfile

U = get_user_model()

# اگر قبلاً جایی رجیستر شده، اول unregister
try:
    admin.site.unregister(U)
except admin.sites.NotRegistered:
    pass

class UserProfileInline(admin.StackedInline):
    model = UserProfile
    can_delete = False
    extra = 0
    fields = ("display_name", "avatar_key", "avatar_preview")
    readonly_fields = ("avatar_preview",)

    def avatar_preview(self, obj):
        key = getattr(obj, "avatar_key", None) or DEFAULT_AVATAR_FALLBACK
        src = DEFAULT_AVATARS.get(key)
        if not src:
            return f"(no preview for key: {key})"
        return mark_safe(
            f'<img src="{src}" width="80" style="border-radius:6px;border:1px solid #ddd;" />'
        )
    avatar_preview.short_description = "Avatar Preview"

# اکشن‌ها را به همین کلاس اضافه می‌کنیم
@admin.register(U)
class UserAdmin(admin.ModelAdmin):
    list_display = ("id", "email", "username", "is_staff", "is_active", "avatar_key_display")
    list_filter = ("is_active", "is_staff", "is_superuser")
    search_fields = ("email", "username")
    readonly_fields = ("date_joined",)
    inlines = [UserProfileInline]

    fieldsets = (
        ("Account", {"fields": ("email", "username", "password")}),
        ("Permissions", {"fields": ("is_active", "is_staff", "is_superuser", "groups", "user_permissions")}),
        ("Important dates", {"fields": ("last_login", "date_joined")}),
    )

    def avatar_key_display(self, obj):
        profile = getattr(obj, "profile", None)
        return getattr(profile, "avatar_key", None) or DEFAULT_AVATAR_FALLBACK
    avatar_key_display.short_description = "Avatar Key"

    

@admin.register(EmailVerification)
class EmailVerificationAdmin(admin.ModelAdmin):
    list_display = ("email", "otp_code", "is_used", "expires_at")
    list_filter = ("is_used",)
    search_fields = ("email", "otp_code")
    ordering = ("-expires_at",)

@admin.register(ReferralBonusConfig)
class ReferralBonusConfigAdmin(admin.ModelAdmin):
    list_display = ("bonus_percentage", "updated_at")
    ordering = ("-updated_at",)
    search_fields = ("bonus_percentage",)