from django.conf import settings
from django.contrib.auth import get_user_model
from django.core.mail import EmailMultiAlternatives
from django.db.models.signals import post_save
from django.dispatch import receiver
from django.template.loader import render_to_string

from .models import Ticket, TicketMessage

User = get_user_model()


@receiver(post_save, sender=TicketMessage)
def notify_user_on_staff_reply(sender, instance: TicketMessage, created, **kwargs):
    # فقط وقتی پیام جدید ایجاد شد و از طرف استاف بود ایمیل بفرست
    if not created or not instance.from_staff:
        return
    ticket = instance.ticket
    user = ticket.user
    subject = f"[Support] Response to your ticket #{ticket.pk}: {ticket.subject}"
    context = {
        "user": user,
        "ticket": ticket,
        "message": instance,
        "site_name": getattr(settings, "SITE_NAME", "RZ Mining"),
    }
    html = render_to_string("email/support_reply.html", context)
    text = f"Hello {getattr(user, 'username', '')},\n\nSupport replied:\n{instance.body}\n\nTicket #{ticket.pk}"

    email = EmailMultiAlternatives(subject=subject, body=text, to=[user.email])
    email.attach_alternative(html, "text/html")
    try:
        email.send(fail_silently=True)  # فعلاً نرم؛ بعداً logging اضافه می‌کنیم
    except Exception:
        pass  # در گام‌های بعدی: logger


@receiver(post_save, sender=Ticket)
def notify_admins_on_new_ticket(sender, instance: Ticket, created, **kwargs):
    if not created:
        return

    # فقط وقتی تیکت جدید ساخته میشه
    admins = User.objects.filter(is_staff=True, is_active=True).values_list("email", flat=True)
    emails = [e for e in admins if e]

    if not emails:
        return

    subject = f"[New Ticket] #{instance.pk}: {instance.subject}"
    context = {
        "ticket": instance,
        "user": instance.user,
        "site_name": getattr(settings, "SITE_NAME", "RZ Mining"),
    }
    html = render_to_string("email/new_ticket_admin.html", context)
    text = f"New support ticket from {instance.user}:\n\nSubject: {instance.subject}\nCategory: {instance.category}\nPriority: {instance.priority}"

    email = EmailMultiAlternatives(subject=subject, body=text, to=emails)
    email.attach_alternative(html, "text/html")
    try:
        email.send(fail_silently=True)
    except Exception:
        pass
