from django.contrib import admin
from django.urls import reverse, NoReverseMatch
from django.utils.html import format_html
from django.utils.timezone import localtime

from .models import Stake


@admin.register(Stake)
class StakeAdmin(admin.ModelAdmin):
    list_display = (
        "id",
        "user_email",
        "miner_link",
        "token_name",
        "amount_formatted",
        "created_at_local",
        "is_active",
    )
    list_filter = ("token", "created_at", "miner", "is_active", "auto_paused")
    search_fields = ("user__email", "miner__name", "token__name", "token__symbol")
    readonly_fields = ("created_at",)
    ordering = ("-created_at",)
    list_per_page = 50
    list_select_related = ("user", "miner", "token")

    # فیلدهای واقعی مدل Stake (هاردکد قبلی غلط بود)
    fields = (
        "user",
        "miner",
        "token",
        "amount",
        "is_active",
        "auto_paused",
        "auto_pause_reason",
        "last_balance_check_at",
        "paused_at",
        "last_accrual_at",
        "last_calculated_at",
        "pending_reward",
        "last_rate_hash",
        "created_at",
    )

    def get_readonly_fields(self, request, obj=None):
        # created_at همیشه read-only بماند
        base = ("created_at",)
        return base

    # ------------- لینک‌های ایمن ادمینی (بدون هاردکد) -------------

    def _admin_change_url(self, obj):
        """دریافت URL ویرایش ادمینی برای هر آبجکت جنگویی به‌صورت داینامیک."""
        if not obj:
            return None
        app_label = obj._meta.app_label
        model_name = obj._meta.model_name
        try:
            return reverse(f"admin:{app_label}_{model_name}_change", args=[obj.pk])
        except NoReverseMatch:
            return None

    def user_email(self, obj):
        user = getattr(obj, "user", None)
        if not user:
            return "-"
        url = self._admin_change_url(user)
        if url:
            return format_html('<a href="{}">{}</a>', url, user.email)
        return user.email  # اگر ادمین رجیستر نشده بود، فقط ایمیل را نشان بده
    user_email.short_description = "User Email"
    user_email.admin_order_field = "user__email"

    def miner_link(self, obj):
        miner = getattr(obj, "miner", None)
        if not miner:
            return "-"
        url = self._admin_change_url(miner)
        if url:
            return format_html('<a href="{}">{}</a>', url, miner.name)
        return miner.name
    miner_link.short_description = "Miner"
    miner_link.admin_order_field = "miner__name"

    def token_name(self, obj):
        token = getattr(obj, "token", None)
        if not token:
            return "-"
        # اگر خواستی برای توکن هم لینک بگذاری، همین الگو را استفاده کن:
        # url = self._admin_change_url(token)
        # if url: return format_html('<a href="{}">{}</a>', url, token.name or token.symbol)
        return getattr(token, "name", None) or getattr(token, "symbol", "-")
    token_name.short_description = "Token"
    token_name.admin_order_field = "token__name"

    def amount_formatted(self, obj):
        # نمایش دقیق تا 8 رقم اعشار
        try:
            return f"{obj.amount:.8f}"
        except Exception:
            return str(obj.amount)
    amount_formatted.short_description = "Amount"

    def created_at_local(self, obj):
        return localtime(obj.created_at).strftime("%Y-%m-%d %H:%M:%S")
    created_at_local.short_description = "Created At"
    created_at_local.admin_order_field = "created_at"