# apps/public/tests/test_public_live_stats.py
from decimal import Decimal

from rest_framework.test import APIClient

from django.contrib.auth import get_user_model
from django.test import TestCase

from apps.miners.models import Miner
from apps.plans.models import Plan
from apps.stakes.models import Stake
from apps.token_app.models import Token


class LiveStatsAPITest(TestCase):
    def setUp(self):
        self.client = APIClient()

        # Token با قیمت دلاری (برای محاسبه USD)
        self.token = Token.objects.create(symbol="MGC", usd_price=Decimal("2.50"), decimals=18)

        # Plan با فیلدهای لازم (بدون token=...)
        self.plan = Plan.objects.create(
            name="Landing Plan",
            level=999,
            price=Decimal("100"),
            power=1,
            monthly_reward_percent=Decimal("4.5"),
        )
        if hasattr(self.plan, "tokens"):
            self.plan.tokens.add(self.token)

        # Miner آنلاین + اتصال به توکن
        self.miner = Miner.objects.create(
            plan=self.plan,
            name="Miner Landing",
            power=1,
            is_online=True,
        )
        if hasattr(self.miner, "tokens"):
            self.miner.tokens.add(self.token)

        # یک کاربر و یک stake برای اینکه اعداد معنی‌دار باشند
        User = get_user_model()
        self.user = User.objects.create_user(
            email="user@test.local",
            username="user1",
        )
        Stake.objects.create(
            user=self.user,
            miner=self.miner,
            token=self.token,
            amount=Decimal("1000"),
            is_active=True,
        )

    def test_live_stats_ok(self):
        resp = self.client.get("/api/public/live-stats/")
        self.assertEqual(resp.status_code, 200)
        data = resp.json()

        # وجود کلیدهای اصلی
        for key in [
            "live",
            "updated_at",
            "online_miners",
            "total_users",
            "total_staked_usd",
            "total_paid_usd",
            "breakdown",
        ]:
            self.assertIn(key, data)

        # شکل breakdown
        self.assertIn("staked", data["breakdown"])
        self.assertIsInstance(data["breakdown"]["staked"], list)
