# plan/views.py

# django
from drf_yasg import openapi

# drf-yasg
from drf_yasg.utils import swagger_auto_schema

# drf
from rest_framework import status, viewsets
from rest_framework.decorators import api_view, permission_classes
from rest_framework.permissions import AllowAny, IsAuthenticated
from rest_framework.response import Response
from rest_framework.throttling import ScopedRateThrottle
from rest_framework.views import APIView

from django.utils.decorators import method_decorator
from django.views.decorators.cache import cache_page

from apps.token_app.models import Token

# local
from .models import Plan
from .serializers import PlanPublicSerializer, PlanSerializer


class PlanViewSet(viewsets.ModelViewSet):
    queryset = Plan.objects.all()
    serializer_class = PlanSerializer

    def destroy(self, request, *args, **kwargs):
        plan = self.get_object()
        plan.delete()
        return Response(status=status.HTTP_204_NO_CONTENT)


@api_view(["POST"])
@permission_classes([IsAuthenticated])
def resolve_plan_by_amount(request):
    try:
        amount = float(request.data.get("amount"))
        token_symbol = request.data.get("token")
    except (TypeError, ValueError):
        return Response({"detail": "Invalid amount or token"}, status=400)

    if amount <= 0:
        return Response({"detail": "Amount must be greater than 0"}, status=400)

    if not token_symbol:
        return Response({"detail": "Token is required"}, status=400)

    token = Token.objects.filter(symbol__iexact=token_symbol).first()
    if not token:
        return Response({"detail": "Invalid token"}, status=400)

    matched_plan = Plan.objects.filter(tokens=token, price__lte=amount).order_by("-price").first()

    if not matched_plan:
        return Response({"detail": "No suitable plan found"}, status=404)

    return Response(PlanSerializer(matched_plan, context={"request": request}).data)


@method_decorator(cache_page(60), name="get")
class PlanListPublicView(APIView):
    permission_classes = [AllowAny]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "public"

    @swagger_auto_schema(
        operation_summary="List available plans (public)",
        operation_description="Returns all plans with monthly_reward_percent and optional video_url.",
        responses={200: "OK"},
    )
    def get(self, request, *args, **kwargs):
        qs = Plan.objects.all().order_by("id")
        ser = PlanPublicSerializer(qs, many=True)
        return Response(ser.data, status=status.HTTP_200_OK)


@method_decorator(cache_page(60), name="get")
class PlanDetailPublicView(APIView):
    permission_classes = [AllowAny]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "public"

    @swagger_auto_schema(
        operation_summary="Retrieve a plan (public)",
        manual_parameters=[
            openapi.Parameter(
                "plan_id", openapi.IN_PATH, description="Plan ID", type=openapi.TYPE_INTEGER
            ),
        ],
        responses={200: "OK", 404: "Not found"},
    )
    def get(self, request, plan_id, *args, **kwargs):
        try:
            obj = Plan.objects.get(id=plan_id)
        except Plan.DoesNotExist:
            return Response({"detail": "Not found"}, status=status.HTTP_404_NOT_FOUND)
        ser = PlanPublicSerializer(obj)
        return Response(ser.data, status=status.HTTP_200_OK)
