from rest_framework import permissions, status, viewsets
from rest_framework.decorators import action, api_view, permission_classes
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response


from django.shortcuts import get_object_or_404
from django.utils import timezone

from .models import (
    EnergyTurboDefinition,
    MinerAccessDefinition,
    Task,
    UserEnergyBoost,
    UserTemporaryMinerAccess,
)
from .serializers import (
    EnergyBoostSerializer,
    MinerAccessSerializer,
    TaskSerializer,
    EventsCatalogSerializer,
)
from .services import _activate_energy_turbo, _activate_miner_access, complete_weekly_restake_task
from .utils import perform_restake


from rest_framework.decorators import api_view, permission_classes
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response
from rest_framework import status
from apps.claims.models import ClaimPurpose
from apps.claims.services import make_claim_signature
from .serializers import RestakeInitiateSerializer

from decimal import Decimal
from rest_framework.decorators import api_view, permission_classes
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response
from rest_framework import status

from apps.claims.services import make_claim_signature
class TaskViewSet(viewsets.ReadOnlyModelViewSet):
    permission_classes = [permissions.IsAuthenticated]
    queryset = Task.objects.filter(is_active=True)
    serializer_class = TaskSerializer

    @action(detail=True, methods=["post"], url_path="claim-miner-access")
    def claim_miner_access(self, request, pk=None):
        """
        اگر تسک از نوع Miner Access باشد ولی reward_object تعیین نشده،
        کاربر می‌تواند با ارسال 'definition_id' یکی از MinerAccessDefinition را انتخاب کند.
        """
        task = self.get_object()
        if task.reward_type != "miner_access":
            return Response({"detail": "not a miner access task"}, status=400)

        definition_id = request.data.get("definition_id")
        if not definition_id:
            return Response({"detail": "definition_id required"}, status=400)

        defn = get_object_or_404(MinerAccessDefinition, id=definition_id, is_active=True)
        access = _activate_miner_access(request.user, defn)
        return Response(MinerAccessSerializer(access).data, status=201)


class RewardViewSet(viewsets.ViewSet):
    permission_classes = [permissions.IsAuthenticated]

    @action(detail=False, methods=["post"], url_path="activate-energy")
    def activate_energy(self, request):
        """
        کاربر می‌تواند Energy Turbo را با یک definition مشخص فعال کند.
        اگر می‌خواهید فقط براساس تسک فعال شود، این endpoint را محدود کنید یا در ادمین پنهان کنید.
        """
        definition_id = request.data.get("definition_id")
        if not definition_id:
            return Response({"detail": "definition_id required"}, status=400)

        defn = get_object_or_404(EnergyTurboDefinition, id=definition_id, is_active=True)
        boosts = _activate_energy_turbo(request.user, defn)
        data = EnergyBoostSerializer(boosts, many=True).data
        return Response(data, status=201)

    @action(detail=False, methods=["get"], url_path="active")
    def active_rewards(self, request):
        now = timezone.now()
        eb = UserEnergyBoost.objects.filter(user=request.user, is_active=True, ends_at__gte=now)
        ma = UserTemporaryMinerAccess.objects.filter(
            user=request.user, is_active=True, ends_at__gte=now
        )
        return Response(
            {
                "energy_boosts": EnergyBoostSerializer(eb, many=True).data,
                "miner_accesses": MinerAccessSerializer(ma, many=True).data,
            }
        )


#
# @api_view(["POST"])
# @permission_classes([IsAuthenticated])
# def restake_view(request):
#     """
#     POST /api/events/restake/
#     {
#       "symbol": "RZ"
#     }
#     """
#     symbol = request.data.get("symbol")
#     if not symbol:
#         return Response({"detail": "symbol required"}, status=status.HTTP_400_BAD_REQUEST)
#
#     try:
#         result = perform_restake(request.user, symbol)
#         complete_weekly_restake_task(request.user)
#         return Response({"success": True, "data": result}, status=200)
#     except Exception as e:
#         return Response({"success": False, "error": str(e)}, status=400)

@api_view(["GET"])
# @permission_classes([IsAuthenticated])
def events_catalog_view(request):
    """
    برمی‌گرداند لیست کل ایونت‌های تعریف‌شده (نه یوزر-اسپسیفیک):

    - tasks: همه‌ی Task های فعال (تعریف شده در admin)
    - energy_turbos: همه‌ی EnergyTurboDefinition های فعال
    - miner_accesses: همه‌ی MinerAccessDefinition های فعال
    """

    tasks = Task.objects.filter(is_active=True).order_by("id")
    energy_defs = EnergyTurboDefinition.objects.filter(is_active=True).order_by("id")
    miner_defs = MinerAccessDefinition.objects.filter(is_active=True).order_by("id")

    serializer = EventsCatalogSerializer(
        {
            "tasks": tasks,
            "energy_turbos": energy_defs,
            "miner_accesses": miner_defs,
        }
    )

    return Response(serializer.data, status=200)


@api_view(["POST"])
@permission_classes([IsAuthenticated])
def restake_initiate_view(request):
    """
    POST /api/events/restake/initiate/
    {
        "miner_id": 123,
        "symbol": "RZ"
    }

    Response:
    {
        "success": true,
        "tx": { ...tx payload from make_claim_signature... }
    }
    """
    serializer = RestakeInitiateSerializer(data=request.data, context={"request": request})
    serializer.is_valid(raise_exception=True)

    user = request.user
    miner = serializer.validated_data["miner"]   # UserMiner
    symbol = serializer.validated_data["symbol"]

    try:
        # amount_requested = None → کل pending کاربر برای این سمبل
        tx_payload = make_claim_signature(
            user,
            symbol,
            None,
            purpose=ClaimPurpose.RESTAKE,
            target_miner=miner,
        )
    except ValueError as e:
        return Response(
            {"success": False, "detail": str(e)},
            status=status.HTTP_400_BAD_REQUEST,
        )
    except Exception:
        return Response(
            {"success": False, "detail": "Internal error"},
            status=status.HTTP_500_INTERNAL_SERVER_ERROR,
        )

    return Response(
        {
            "success": True,
            "tx": tx_payload,
        },
        status=status.HTTP_200_OK,
    )




@api_view(["POST"])
@permission_classes([IsAuthenticated])
def restake_sign_view(request):
    """
    POST /api/events/restake/sign/
    {
      "symbol": "RZ"
    }

    خروجی: همون payload امضایی که ClaimSignView برمی‌گردونه،
    فقط ما ازش برای ری‌استیک استفاده می‌کنیم.
    """
    user = request.user
    data = request.data or {}
    symbol = data.get("symbol")

    if not symbol:
        return Response({"detail": "symbol_required"}, status=status.HTTP_400_BAD_REQUEST)

    try:
        # amount_requested = None یعنی "کل pending" همون سمبل
        payload = make_claim_signature(user, symbol, None)
    except ValueError as e:
        # همون پیام‌های استانداردی که الان داری: wallet_not_connected, unknown_token, nothing_to_claim, bad_amount, ...
        return Response({"detail": str(e)}, status=status.HTTP_400_BAD_REQUEST)

    return Response(
        {
            "success": True,
            "tx": payload,
        },
        status=status.HTTP_200_OK,
    )