from decimal import Decimal
from uuid import uuid4

from django.db import transaction
from django.utils import timezone

from apps.claims.models import ClaimTx
from apps.claims.services import get_pending_by_symbol_for_user
from apps.stakes.models import Stake
from apps.token_app.models import Token
from apps.wallets.models import WalletConnection  # اگر داشتی برای آدرس کاربر

@transaction.atomic
def perform_restake(user, symbol: str) -> dict:
    """
    کل سود معوق کاربر برای 'symbol' را به استیک فعال همان توکن اضافه می‌کند
    و برای صفر شدن pending، یک ClaimTx داخلی (synthetic) ثبت می‌کند.
    عملیات اتمیک است.
    """
    # ۱) توکن
    token = Token.objects.filter(symbol__iexact=symbol).first()
    if not token:
        raise ValueError("Invalid token symbol")

    # ۲) مقدار pending فعلی
    pending_amount = get_pending_by_symbol_for_user(user, symbol)
    if pending_amount <= 0:
        raise ValueError("No pending earnings to restake")

    # ۳) استیک فعال قفل شود
    stake = (
        Stake.objects.select_for_update()
        .filter(user=user, token=token, is_active=True)
        .first()
    )
    if not stake:
        raise ValueError("Active stake for this token not found")

    # ۴) افزایش amount استیک
    stake.amount = (Decimal(stake.amount) + Decimal(pending_amount)).quantize(Decimal("0.00000001"))
    stake.save(update_fields=["amount"])

    # ۵) ثبت ClaimTx داخلی برای صفر شدن pending
    decs = int(getattr(token, "decimals", 18) or 18)
    amount_scaled = int(Decimal(pending_amount) * (Decimal(10) ** Decimal(decs)))

    user_wallet = (
        WalletConnection.objects.filter(user=user)
        .values_list("wallet_address", flat=True)
        .order_by("-id")
        .first()
    ) or ""

    token_address = (
        getattr(token, "contract_address", None)
        or getattr(token, "token_address", None)
        or ""
    )

    ClaimTx.objects.create(
        user=user,
        user_address=user_wallet,
        token_address=token_address,
        token_symbol=token.symbol,
        amount_scaled=Decimal(amount_scaled),
        nonce=0,
        tx_hash=f"restake:{uuid4().hex}",  # غیرزنجیره‌ای، فقط مارکر داخلی
    )

    return {
        "token": token.symbol,
        "restaked_amount": str(pending_amount),
        "new_stake_amount": str(stake.amount),
        "ts": timezone.now().isoformat(),
    }