# apps/claims/views.py
import gzip
import hashlib
import hmac
import json
import logging
from collections.abc import Iterable
from decimal import Decimal
from typing import Any
from apps.events.services import apply_restake_after_claim
from rest_framework import status
from rest_framework.permissions import AllowAny, IsAuthenticated
from rest_framework.response import Response
from rest_framework.views import APIView

from django.conf import settings
from django.db import transaction
from django.http import JsonResponse
from django.utils import timezone

from apps.claims.models import ClaimIntent, ClaimTx
from apps.token_app.models import Token
from apps.wallets.models import WalletConnection

from .services import make_claim_signature
from .utils import parse_claimed_log

logger = logging.getLogger(__name__)

CHAIN_ID = int(getattr(settings, "CHAIN_ID", 56))
CLAIM_DEADLINE_SEC = 5 * 60  # 5 دقیقه


# ---------- همون دو هِلپر پارس قبلی توی فایل تو موجود بود؛ اگر جدا داری، از همونا استفاده کن ----------
def _iter_possible_events(payload: dict) -> Iterable[dict]:
    """
    ورودی‌های محتمل از QuickNode:
      - { "event": {...} }
      - { "events": [ {...}, ... ] }
      - { "data": { "event": {...} } } یا { "data": { "events": [...] } }
      - { "matchingReceipts": [ {...receipt-like...} ] }   # ← Streams: matchingReceipts
      - { "matchingLogs": [ {...log-like...} ] }           # ← بعضی فیدها فقط لاگ می‌دهند
      - فرم فلت (fallback) با کلیدهای user_address/token_address/...
    خروجی هر آیتم یک «event-like dict» با کلیدهای حداقل:
      { "transaction": {"hash": ...}, "logs": [ {"decoded": {...}} ] }
    """
    if not isinstance(payload, dict):
        return []

    # 1) مستقیم
    if isinstance(payload.get("event"), dict):
        return [payload["event"]]
    if isinstance(payload.get("events"), list):
        return payload["events"]

    # 2) داخل data
    data = payload.get("data")
    if isinstance(data, dict):
        if isinstance(data.get("event"), dict):
            return [data["event"]]
        if isinstance(data.get("events"), list):
            return data["events"]

    # 3) QuickNode Streams: matchingReceipts
    # معمولاً هر receipt همین شکل مورد انتظار ما را دارد:
    # { "transaction": {"hash": ...}, "logs": [ {"decoded": {...}} , ...] }
    mr = payload.get("matchingReceipts")
    if isinstance(mr, list) and mr:
        # اگر آیتم‌ها خودشان receipt کامل هستند، همان را برگردان
        out = []
        for it in mr:
            if isinstance(it, dict):
                # اگر ساختار لاگ/دی‌کد شده داخل نیست، دست‌کم wrap کنیم
                if "transaction" in it and "logs" in it:
                    out.append(it)
                else:
                    # حداقلی: تبدیل به event سازگار
                    txh = (
                        (it.get("transaction") or {}).get("hash")
                        or it.get("hash")
                        or it.get("txHash")
                        or ""
                    )
                    logs = it.get("logs") or []
                    out.append({"transaction": {"hash": txh}, "logs": logs})
        if out:
            return out

    # 4) QuickNode Streams: matchingLogs (لاگ‌های تکی بدون receipt)
    ml = payload.get("matchingLogs")
    if isinstance(ml, list) and ml:
        out = []
        for lg in ml:
            if isinstance(lg, dict):
                txh = lg.get("transactionHash") or lg.get("hash") or ""
                out.append(
                    {
                        "transaction": {"hash": txh},
                        "logs": [lg],  # یک لاگ را به آرایه‌ی logs تبدیل می‌کنیم
                    }
                )
        if out:
            return out

    # 5) فرم فلت مینیمال (fallback)
    minimal = {"user_address", "token_address", "amount_scaled", "nonce", "tx_hash"}
    if any(k in payload for k in minimal):
        return [
            {
                "transaction": {"hash": payload.get("tx_hash")},
                "logs": [
                    {
                        "decoded": {
                            "name": "Claimed",
                            "params": [
                                {"name": "user", "value": payload.get("user_address")},
                                {"name": "token", "value": payload.get("token_address")},
                                {"name": "amount", "value": payload.get("amount_scaled")},
                                {"name": "nonce", "value": payload.get("nonce")},
                            ],
                        }
                    }
                ],
            }
        ]

    return []


def _extract_claims_from_event(ev: dict) -> list[dict]:
    out = []
    tx_hash = ((ev or {}).get("transaction") or {}).get("hash") or ""
    logs = (ev or {}).get("logs") or []
    for lg in logs:
        dec = (lg or {}).get("decoded") or {}
        if (dec.get("name") or "").lower() != "claimed":
            continue
        params = dec.get("params") or []
        pmap = {
            (p.get("name") or "").lower(): p.get("value") for p in params if isinstance(p, dict)
        }
        user = pmap.get("user") or pmap.get("account") or pmap.get("useraddress")
        token = pmap.get("token") or pmap.get("tokenaddress")
        amount = pmap.get("amount")
        nonce = pmap.get("nonce")
        if user and token and amount is not None and nonce is not None:
            out.append(
                {
                    "user_address": str(user),
                    "token_address": str(token),
                    "amount_scaled": str(amount),
                    "nonce": int(nonce),
                    "tx_hash": str(tx_hash or lg.get("transactionHash") or ""),
                }
            )
    return out


def parse_quicknode_payload(body: dict[str, Any]) -> tuple[dict[str, Any] | None, list]:
    errors = []
    flat_keys = ["user_address", "token_address", "amount_scaled", "nonce", "tx_hash"]
    if all(k in body for k in flat_keys):
        try:
            data = {
                "user_address": _lower_hex(body.get("user_address")),
                "token_address": _lower_hex(body.get("token_address")),
                "amount_scaled": _as_str(body.get("amount_scaled")),
                "nonce": int(body.get("nonce")),
                "tx_hash": _as_str(body.get("tx_hash")),
            }
            return data, errors
        except Exception as e:
            errors.append(f"flat_parse_error: {e}")
    event_obj = body.get("event")
    if event_obj is None:
        data_arr = body.get("data") or []
        for it in data_arr:
            if isinstance(it, dict) and it.get("event"):
                event_obj = it.get("event")
                break
    if event_obj:
        tx_obj = event_obj.get("transaction") or {}
        tx_hash = _as_str(tx_obj.get("hash"))
        logs = event_obj.get("logs") or []
        first_log = None
        for lg in logs:
            dec = lg.get("decoded")
            if isinstance(dec, dict) and (
                dec.get("name") in ["Claimed", "claimed", "Claim", "WithdrawnClaim"]
            ):
                first_log = lg
                break
        if not first_log:
            for lg in logs:
                if isinstance(lg.get("decoded"), dict):
                    first_log = lg
                    break
        if not first_log:
            return None, ["no_decoded_log_found"]
        decoded = first_log.get("decoded") or {}
        params = decoded.get("params") or []
        by_name = {}
        for p in params:
            n = p.get("name")
            if n:
                by_name[n] = p.get("value")
        user_address = by_name.get("user") or by_name.get("account") or by_name.get("to") or ""
        token_address = by_name.get("token") or by_name.get("tokenAddress") or ""
        amount_val = by_name.get("amount") or by_name.get("value") or by_name.get("amt")
        nonce_val = by_name.get("nonce") or by_name.get("salt") or by_name.get("id")
        if (
            not user_address
            or not token_address
            or amount_val is None
            or nonce_val is None
            or not tx_hash
        ):
            try:
                v0 = params[0].get("value") if len(params) > 0 else None
                v1 = params[1].get("value") if len(params) > 1 else None
                v2 = params[2].get("value") if len(params) > 2 else None
                v3 = params[3].get("value") if len(params) > 3 else None
                user_address = user_address or v0 or ""
                token_address = token_address or v1 or ""
                amount_val = amount_val if amount_val is not None else v2
                nonce_val = nonce_val if nonce_val is not None else v3
            except Exception:
                pass
        if not (
            user_address
            and token_address
            and amount_val is not None
            and nonce_val is not None
            and tx_hash
        ):
            return None, ["missing_required_fields_from_decoded"]
        try:
            data = {
                "user_address": _lower_hex(user_address),
                "token_address": _lower_hex(token_address),
                "amount_scaled": _as_str(amount_val),
                "nonce": int(_as_str(nonce_val)),
                "tx_hash": tx_hash,
            }
            return data, errors
        except Exception as e:
            return None, [f"decode_cast_error: {e}"]
    return None, ["no_event_object"]


def handle_claim_delivery(
    *, user_address: str, token_address: str, amount_scaled: str, nonce: int, tx_hash: str
) -> bool:
    tok = (
        Token.objects.filter(contract_address__iexact=token_address).first()
        or Token.objects.filter(symbol__iexact="MGC").first()
    )
    token_symbol = getattr(tok, "symbol", None) or "UNKNOWN"
    from apps.wallets.models import WalletConnection

    u = (
        WalletConnection.objects.filter(wallet_address__iexact=user_address)
        .values_list("user_id", flat=True)
        .first()
    )

    with transaction.atomic():
        ClaimTx.objects.get_or_create(
            tx_hash=tx_hash,
            defaults={
                "user_id": u,
                "user_address": user_address,
                "token_address": token_address,
                "token_symbol": token_symbol,
                "amount_scaled": Decimal(str(amount_scaled)),
                "nonce": int(nonce),
            },
        )
    from apps.claims.models import ClaimIntent

    ClaimIntent.objects.filter(user_address__iexact=user_address, nonce=nonce).update(
        status="confirmed", updated_at=timezone.now()
    )

    return True


def _bytes(x) -> bytes:
    if isinstance(x, bytes):
        return x
    if isinstance(x, str):
        return x.encode("utf-8")
    return json.dumps(x, separators=(",", ":"), ensure_ascii=False).encode("utf-8")


def _verify_qn_signature(
    secret: str, nonce: str, timestamp: str, raw_payload: bytes, given_sig_hex: str
) -> bool:
    """
    QuickNode: HMAC-SHA256(secret, nonce + timestamp + raw_body)
    """
    if not (secret and nonce and timestamp and given_sig_hex):
        return False
    message = (nonce + timestamp).encode("utf-8") + (raw_payload or b"")
    computed = hmac.new(_bytes(secret), message, hashlib.sha256).hexdigest()
    try:
        return hmac.compare_digest(computed, given_sig_hex.lower())
    except Exception:
        return False


def _as_str(x) -> str:
    return "" if x is None else str(x)


def _lower_hex(addr: str | None) -> str:
    return (addr or "").strip().lower()


def _extract_from_decoded_log(ev: dict) -> list[dict]:
    """
    وقتی QuickNode log را به صورت decoded می‌دهد:
    ev => {"transaction":{"hash":...},"logs":[{"decoded":{"name":"Claimed","params":[...]}}]}
    """
    out = []
    tx_hash = ((ev or {}).get("transaction") or {}).get("hash") or ""
    logs = (ev or {}).get("logs") or []
    for lg in logs:
        dec = (lg or {}).get("decoded") or {}
        name = (dec.get("name") or "").lower()
        if name not in ("claimed", "withdrawn", "claim", "withdraw"):
            continue
        params = dec.get("params") or []
        pmap = {
            (p.get("name") or "").lower(): p.get("value") for p in params if isinstance(p, dict)
        }
        user = pmap.get("user") or pmap.get("to") or pmap.get("account") or pmap.get("useraddress")
        token = pmap.get("token") or pmap.get("tokenaddress")
        amount = pmap.get("amount") or pmap.get("value") or pmap.get("amt")
        nonce = pmap.get("nonce") or pmap.get("salt") or pmap.get("id")
        if user and token and amount is not None and nonce is not None:
            out.append(
                {
                    "user_address": _lower_hex(user),
                    "token_address": _lower_hex(token),
                    "amount_scaled": _as_str(amount),
                    "nonce": int(_as_str(nonce)),
                    "tx_hash": str(tx_hash or lg.get("transactionHash") or ""),
                }
            )
    return out


def _iter_candidate_events(body: Any) -> Iterable[dict]:
    """
    بدنه را normalize می‌کند تا لیست event/receipt بدهد.
    پوشش می‌دهد:
      - {"event": {...}}
      - {"events": [ {...}, ... ]}
      - {"data": [ {"event": {...}}, ... ]} / {"data": {"event": {...}}}
      - {"logs":[...], "transaction":{...}}  (logs-top-level)
      - {"matchingReceipts":[ {...}, ... ]}  (QuickNode receipts stream)
      - log یا receipt منفرد با topics+data
    """
    if isinstance(body, dict):
        # 1) مستقیم event/ events
        if isinstance(body.get("event"), dict):
            yield body["event"]
            return
        if isinstance(body.get("events"), list):
            for ev in body["events"]:
                if isinstance(ev, dict):
                    yield ev
            return

        # 2) data → event
        data = body.get("data")
        if isinstance(data, list):
            for it in data:
                if isinstance(it, dict) and isinstance(it.get("event"), dict):
                    yield it["event"]
            # اگر ساختار receipt هم در ریشه هست
            if body.get("logs") or body.get("transaction"):
                yield body
            return
        if isinstance(data, dict) and isinstance(data.get("event"), dict):
            yield data["event"]
            return

        # 3) receipts stream
        if isinstance(body.get("matchingReceipts"), list):
            for rec in body["matchingReceipts"]:
                if isinstance(rec, dict):
                    # خود receipt را بده؛ extractor ها logs/transactionHash را برمی‌دارند
                    yield rec
            return

        # 4) logs/transaction در ریشه
        if body.get("logs") and (body.get("transaction") or body.get("transactionHash")):
            yield body
            return

        # 5) یک log/receipt منفرد
        if body.get("topics") and body.get("data"):
            yield body
            return

    elif isinstance(body, list):
        for it in body:
            if isinstance(it, dict):
                if it.get("event") and isinstance(it["event"], dict):
                    yield it["event"]
                else:
                    yield it
        return

    return []  # nothing recognized


def _extract_from_topics_log(ev_or_log: dict) -> list[dict]:
    """
    وقتی فقط topics+data داریم (یا receipt با logs)، از util موجود استفاده می‌کنیم.
    از receipt هم tx_hash را درست برمی‌داریم.
    """
    # receipt-Style: ممکن است در سطح بالا transactionHash داشته باشیم
    receipt_tx = _as_str(
        (ev_or_log or {}).get("transactionHash")
        or ((ev_or_log or {}).get("transaction") or {}).get("hash")
        or (ev_or_log or {}).get("hash")
        or ""
    )

    # حالت event با logs:
    logs = []
    if "logs" in (ev_or_log or {}):
        logs = ev_or_log.get("logs") or []
    else:
        # شاید خود log منفرد باشد
        if (ev_or_log or {}).get("topics") and (ev_or_log or {}).get("data"):
            logs = [ev_or_log]

    out = []
    for lg in logs:
        try:
            parsed = parse_claimed_log(lg)  # خروجی: tx_hash, user, token, amount_scaled, nonce
        except Exception:
            parsed = None
        if not parsed:
            continue

        tx_hash = _as_str(parsed.get("tx_hash") or lg.get("transactionHash") or receipt_tx)
        out.append(
            {
                "user_address": _lower_hex(parsed["user"]),
                "token_address": _lower_hex(parsed["token"]),
                "amount_scaled": _as_str(parsed["amount_scaled"]),
                "nonce": int(parsed["nonce"]),
                "tx_hash": tx_hash,
            }
        )
    return out


def _persist_claim(
    user_address: str, token_address: str, amount_scaled: str | int, nonce: int, tx_hash: str
) -> bool:
    tok = (
        Token.objects.filter(contract_address__iexact=token_address).first()
        or Token.objects.filter(symbol__iexact="MGC").first()
    )
    token_symbol = getattr(tok, "symbol", None) or "UNKNOWN"
    u = (
        WalletConnection.objects.filter(wallet_address__iexact=user_address)
        .values_list("user_id", flat=True)
        .first()
    )

    with transaction.atomic():
        ClaimTx.objects.get_or_create(
            tx_hash=tx_hash,
            defaults={
                "user_id": u,
                "user_address": user_address,
                "token_address": token_address,
                "token_symbol": token_symbol,
                "amount_scaled": Decimal(str(amount_scaled)),
                "nonce": int(nonce),
            },
        )
        # اگر intentی با همین nonce و آدرس کاربر بوده → تاییدش کن
        ClaimIntent.objects.filter(user_address__iexact=user_address, nonce=int(nonce)).update(
            status="confirmed", updated_at=timezone.now()
        )

    return True


class ClaimSignView(APIView):
    permission_classes = [IsAuthenticated]

    def post(self, request):
        symbol = (request.data or {}).get("symbol")
        amount = (request.data or {}).get("amount")

        if not symbol:
            return Response({"detail": "symbol_required"}, status=400)

        try:
            amount_dec = (
                Decimal(amount)
                if amount
                not in (
                    None,
                    "",
                )
                else None
            )
        except Exception:
            return Response({"detail": "bad_amount"}, status=400)

        try:
            payload = make_claim_signature(request.user, symbol, amount_dec)
            return Response(payload, status=200)
        except ValueError as e:
            # خطای منطقی کنترل‌شده
            return Response({"detail": f"claim_error:{e}"}, status=400)
        except Exception:
            # خطای غیرمنتظره → لاگ دقیق
            logger.exception("claim_sign failed")
            return Response({"detail": "server_error"}, status=500)


class QuickNodeClaimedWebhookView(APIView):
    permission_classes = [AllowAny]

    def post(self, request, *args, **kwargs):
        # 0) خواندن raw و بازکردن gzip فقط برای parsing (نه برای امضا)
        raw_body: bytes = request.body or b""
        content_encoding = (request.headers.get("Content-Encoding") or "").lower()

        try:
            payload_text = (
                gzip.decompress(raw_body).decode("utf-8")
                if (content_encoding == "gzip" and raw_body)
                else (raw_body.decode("utf-8") if raw_body else "{}")
            )
        except Exception:
            logger.exception("webhook: gzip decompress failed")
            return JsonResponse({"detail": "bad_payload"}, status=400)

        # 1) اعتبارسنجی امضا (X-QN-*)
        qn_nonce = request.headers.get("X-QN-Nonce")
        qn_timestamp = request.headers.get("X-QN-Timestamp")
        qn_signature = request.headers.get("X-QN-Signature")
        secret = getattr(settings, "QN_WEBHOOK_SECRET", None)

        if qn_nonce and qn_timestamp and qn_signature:
            if not secret:
                logger.error("webhook: missing QN_WEBHOOK_SECRET")
                return JsonResponse({"detail": "server_misconfigured"}, status=500)
            ok = _verify_qn_signature(secret, qn_nonce, qn_timestamp, raw_body, qn_signature)
            if not ok:
                logger.error("webhook: signature verification failed (X-QN-*)")
                return JsonResponse({"detail": "unauthorized"}, status=401)
        else:
            # fallback قدیمی: X-QuickNode-Token
            expected = getattr(settings, "QN_WEBHOOK_SECRET", None)
            got = request.headers.get("X-QuickNode-Token")
            if not expected or got != expected:
                logger.error("webhook: simple token auth failed")
                return JsonResponse({"detail": "unauthorized"}, status=401)

        # 2) parse JSON
        try:
            body = json.loads(payload_text) if payload_text else {}
        except Exception:
            logger.exception("webhook: json parse error")
            return JsonResponse({"detail": "bad_json"}, status=400)

        # 3) normalize → candidate events
        events = list(_iter_candidate_events(body))
        if not events:
            # لاگ تشخیصی با کلیدها
            try:
                root_keys = list(body.keys()) if isinstance(body, dict) else type(body).__name__
            except Exception:
                root_keys = "uninspectable"
            logger.error(
                "quicknode webhook: parse failed: no_event_object; root_keys=%s", root_keys
            )
            return JsonResponse(
                {"ok": True, "deliveries": 1, "handled": 0, "errors": ["no_event_object"]},
                status=200,
            )

        deliveries, handled, errors = 0, 0, []

        for ev in events:
            deliveries += 1
            claims: list[dict] = []

            # مسیر 1: decoded logs
            try:
                claims = _extract_from_decoded_log(ev)
            except Exception as e:
                logger.exception("quicknode webhook: decoded-extract failed")
                errors.append(f"decoded_extract_failed:{e}")

            # مسیر 2: topics+data (اگر decoded چیزی نداد)
            if not claims:
                try:
                    claims = _extract_from_topics_log(ev)
                except Exception as e:
                    logger.exception("quicknode webhook: topics-extract failed")
                    errors.append(f"topics_extract_failed:{e}")

            if not claims:
                # برای دیباگ: نمونه‌ای از ساختار event را لاگ کن (بدون اسپم)
                logger.warning(
                    "quicknode webhook: event had no recognizable claim; keys=%s",
                    list(ev.keys()) if isinstance(ev, dict) else type(ev).__name__,
                )
                continue

            # ذخیره هر claim
            for c in claims:
                try:
                    if _persist_claim(**c):
                        apply_restake_after_claim(
                            user_address=c["user_address"],
                            token_address=c["token_address"],
                            amount_scaled=c["amount_scaled"],
                            nonce=c["nonce"],
                            tx_hash=c["tx_hash"],
                        )
                        handled += 1
                except Exception as e:
                    logger.exception("quicknode webhook: persist failed")
                    errors.append(f"persist_failed:{e}")

        return JsonResponse(
            {"ok": True, "deliveries": deliveries, "handled": handled, "errors": errors}, status=200
        )


class UserClaimIntentsAdminView(APIView):
    """
    Admin-only endpoint to view all claim intents + claim txs for a user.
    Example: GET /api/claims/user-claim-data/21/
    """

    def get(self, request, user_id):
        try:
            # --- ClaimIntent ---
            intents_qs = (
                ClaimIntent.objects.filter(user_id=user_id)
                .order_by("-created_at")
                .values(
                    "id",
                    "user_id",
                    "token_symbol",
                    "token_address",
                    "user_address",
                    "amount_scaled",
                    "nonce",
                    "deadline",
                    "signature",
                    "status",
                    "created_at",
                    "updated_at",
                )
            )
            intents = []
            for i in intents_qs:
                i["tx_hash"] = None  # هنوز تأیید نشده
                intents.append(i)

            # --- ClaimTx ---
            txs_qs = (
                ClaimTx.objects.filter(user_id=user_id)
                .order_by("-created_at")
                .values(
                    "id",
                    "user_id",
                    "user_address",
                    "token_symbol",
                    "token_address",
                    "amount_scaled",
                    "nonce",
                    "tx_hash",
                    "created_at",
                )
            )

            data = {
                "user_id": user_id,
                "claim_intents": intents,
                "claim_txs": list(txs_qs),
            }

            if not intents and not txs_qs.exists():
                return Response(
                    {"detail": "No claim data found for this user."},
                    status=status.HTTP_404_NOT_FOUND,
                )

            return Response(data, status=status.HTTP_200_OK)

        except Exception as e:
            logging.exception("admin-user-claim-data-get failed")
            return Response(
                {"detail": "server_error", "error": str(e)},
                status=status.HTTP_500_INTERNAL_SERVER_ERROR,
            )
