from __future__ import annotations

from typing import Iterable, List

from django.conf import settings
from django.contrib.auth import get_user_model
from django.core.mail import EmailMultiAlternatives
from django.db import transaction
from django.template.loader import render_to_string
from django.utils import timezone

from .models import CampaignStatus, EmailCampaign, EmailDelivery

User = get_user_model()

BATCH_SIZE = getattr(settings, "BROADCAST_EMAIL_BATCH_SIZE", 50)


def _eligible_users() -> Iterable[User]:
    return User.objects.filter(is_active=True).exclude(email__isnull=True).exclude(email="")


@transaction.atomic
def schedule_campaign(campaign: EmailCampaign) -> None:
    """
    وضعیت کمپین را به SCHEDULED می‌برد و EmailDelivery ها را برای همه‌ی کاربران فعال می‌سازد.
    اگر قبلاً ساخته شده باشند، دوباره نمی‌سازد.
    """
    if campaign.status not in [CampaignStatus.DRAFT, CampaignStatus.FAILED]:
        return
    users_qs = _eligible_users().only("id")
    deliveries_to_create: List[EmailDelivery] = []
    existing_user_ids = set(
        EmailDelivery.objects.filter(campaign=campaign).values_list("user_id", flat=True)
    )
    for uid in users_qs.values_list("id", flat=True):
        if uid not in existing_user_ids:
            deliveries_to_create.append(EmailDelivery(campaign=campaign, user_id=uid))

    if deliveries_to_create:
        EmailDelivery.objects.bulk_create(deliveries_to_create, batch_size=1000)

    campaign.total_recipients = EmailDelivery.objects.filter(campaign=campaign).count()
    campaign.sent_count = EmailDelivery.objects.filter(campaign=campaign, sent=True).count()
    campaign.error_count = (
        EmailDelivery.objects.filter(campaign=campaign, sent=False)
        .exclude(error_message__isnull=True)
        .exclude(error_message="")
        .count()
    )
    campaign.status = CampaignStatus.SCHEDULED
    campaign.scheduled_at = timezone.now()
    campaign.save(
        update_fields=["total_recipients", "sent_count", "error_count", "status", "scheduled_at"]
    )


def _render_personalized_html(campaign: EmailCampaign, user: User) -> str:
    """
    Render campaign email using base template and text_content field (no manual HTML input by admin).
    """
    context = {
        "subject": campaign.subject,
        "text_content": campaign.text_content,
        "username": getattr(user, "username", "") or getattr(user, "email", ""),
        "site_name": getattr(settings, "SITE_NAME", "CoinMaining Game"),
        "image_url": campaign.image.url if campaign.image else None,
        "year": timezone.now().year,
    }

    # قالب HTML خودکار از روی template آماده
    return render_to_string("email/emails/templates/base_campaign.html", context)

def _send_one(campaign: EmailCampaign, delivery: EmailDelivery) -> None:
    user = delivery.user
    subject = campaign.subject
    html = _render_personalized_html(campaign, user)
    text = f"Hi {getattr(user, 'username', '')},\n\n"  # fallback text

    email = EmailMultiAlternatives(
        subject=subject,
        body=text,
        from_email=getattr(settings, "DEFAULT_FROM_EMAIL", None),
        to=[user.email],
    )
    email.attach_alternative(html, "text/html")

    # Attach optional files if exist
    if campaign.image:
        try:
            email.attach_file(campaign.image.path)
        except Exception as e:
            print(f"[Broadcast] Could not attach image: {e}")

    if campaign.attachment:
        try:
            email.attach_file(campaign.attachment.path)
        except Exception as e:
            print(f"[Broadcast] Could not attach file: {e}")

    email.send(fail_silently=False)


@transaction.atomic
def send_batch(campaign: EmailCampaign, batch_size: int | None = None) -> int:
    """
    یک batch از ایمیل‌ها را ارسال می‌کند. خروجی: تعداد ارسال‌های موفق در این Batch.
    """
    if campaign.status in [CampaignStatus.COMPLETED, CampaignStatus.FAILED]:
        return 0

    size = batch_size or BATCH_SIZE

    # قفل منطقی: کمپین را در وضعیت SENDING ببریم
    if campaign.status != CampaignStatus.SENDING:
        campaign.status = CampaignStatus.SENDING
        campaign.save(update_fields=["status"])

    # انتخاب گیرنده‌های ارسال‌نشده و بدون خطا
    deliveries = (
        EmailDelivery.objects.select_related("user")
        .filter(campaign=campaign, sent=False, error_message__isnull=True)
        .order_by("id")[:size]
    )

    sent_count_increment = 0
    for d in deliveries:
        try:
            _send_one(campaign, d)
            d.sent = True
            d.sent_at = timezone.now()
            d.save(update_fields=["sent", "sent_at"])
            sent_count_increment += 1
        except Exception as e:
            d.error_message = str(e)[:1000]
            d.save(update_fields=["error_message"])

    # به‌روزرسانی شمارنده‌ها
    campaign.sent_count = EmailDelivery.objects.filter(campaign=campaign, sent=True).count()
    campaign.error_count = (
        EmailDelivery.objects.filter(campaign=campaign, sent=False)
        .exclude(error_message__isnull=True)
        .exclude(error_message="")
        .count()
    )

    # اگر دیگر گیرنده‌ای نداریم، پایان
    remaining = EmailDelivery.objects.filter(
        campaign=campaign, sent=False, error_message__isnull=True
    ).exists()
    if not remaining:
        campaign.status = CampaignStatus.COMPLETED
    campaign.save(update_fields=["sent_count", "error_count", "status"])
    return sent_count_increment


def process_pending_campaigns(
    max_batches_per_campaign: int = 5, batch_size: int | None = None
) -> int:
    """
    برای Cron: کمپین‌های Scheduled/Sending را پردازش می‌کند.
    خروجی: تعداد کل ایمیل‌های ارسال‌شده در این اجرای Cron.
    """
    import time

    total_sent = 0
    now = timezone.now()
    campaigns = EmailCampaign.objects.filter(
        status__in=[CampaignStatus.SCHEDULED, CampaignStatus.SENDING],
    )

    for c in campaigns:
        # اگر scheduled_at در آینده تنظیم شد (برای آینده)، از این اجرا صرف‌نظر کن
        if c.scheduled_at and c.scheduled_at > now:
            continue

        # چند batch پشت‌سر‌هم برای همین کمپین (برای سرعت) ولی با سقف
        for _ in range(max_batches_per_campaign):
            time.sleep(getattr(settings, "BROADCAST_SLEEP_BETWEEN_BATCHES", 0))
            sent = send_batch(c, batch_size=batch_size)
            total_sent += sent
            if sent == 0:  # دیگر چیزی برای ارسال نیست یا به COMPLETED رفت
                break

    return total_sent
