# apps/badges/tests/test_referral_badges.py

# stdlib
from decimal import Decimal
from uuid import uuid4

from django.apps import apps
from django.contrib.auth import get_user_model

# django
from django.test import TestCase

from apps.badges.services import check_and_grant_referral_badges
from apps.miners.models import Miner
from apps.plans.models import Plan
from apps.stakes.models import Stake

# local
from apps.token_app.models import Token
from apps.users.models import ReferralInvite, ReferralRelation
from apps.users.services.referrals import count_active_referrals

U = get_user_model()


def seed_badge(key: str, name: str, reward_bonus_bp: int, image_key: str, description: str = ""):
    """
    Badge را بر اساس مدل فعلی شما seed می‌کند:
    fields: key, name, description, image_key, reward_bonus_bp, is_active
    """
    Badge = apps.get_model("badges", "Badge")
    if not Badge.objects.filter(key=key).exists():
        Badge.objects.create(
            key=key,
            name=name,
            description=description,
            image_key=image_key,
            reward_bonus_bp=reward_bonus_bp,  # basis points (100bp = 1%)
            is_active=True,
        )


class ReferralBadgesTests(TestCase):
    def setUp(self):
        # 0) Seed موردنیاز سرویس «ارجاع»
        # توجه: مقدار bp صرفاً نمونه است (0.05% و 0.10%). اگر سیاست واقعی فرق دارد، عوضش کن.
        seed_badge("referral_5", "Referral 5", 5, "ref5")  # 5 bp = 0.05%
        seed_badge("referral_10", "Referral 10", 10, "ref10")  # 10 bp = 0.10%

        # 1) Token پایه
        self.token = Token.objects.create(
            symbol="MGC",
            name="Magic Coin",
            usd_price=Decimal("1.00"),
            decimals=18,
        )

        # 2) Plan مطابق مدل پروژه
        self.plan = Plan.objects.create(
            name="p-test",
            level=1,
            power=100,
            price=Decimal("10.00"),
            description="plan for tests",
            video_url="",
            monthly_reward_percent=Decimal("4.5"),
        )
        self.plan.tokens.add(self.token)

        # 3) Miner متصل به همین plan
        self.miner = Miner.objects.create(
            name="Miner Test",
            plan=self.plan,
            power=100,
            is_online=True,
        )
        if hasattr(self.miner, "tokens"):
            self.miner.tokens.add(self.token)

        # 4) inviter + invite
        self.inviter = U.objects.create(
            email="inviter@test.local",
            username=f"inviter_{uuid4().hex[:8]}",
            is_active=True,
        )
        self.invite, _ = ReferralInvite.objects.get_or_create(inviter=self.inviter)

    def _add_active_invitee(self):
        """invitee + relation + stake فعال، و فعال‌سازی relation."""
        v = U.objects.create(
            email=f"inv+{uuid4().hex[:10]}@local",
            username=f"user_{uuid4().hex[:8]}",
            is_active=True,
        )
        ReferralRelation.objects.get_or_create(
            inviter=self.inviter,
            invitee=v,
            defaults={"active": False, "invite": self.invite},
        )
        Stake.objects.create(
            user=v,
            miner=self.miner,
            token=self.token,
            amount=Decimal("1000"),
            is_active=True,
        )
        # اگر سیگنال/سرویس آپدیت active ندارید:
        ReferralRelation.objects.filter(invitee=v).update(active=True)
        return v

    def test_no_badge_below_5(self):
        for _ in range(4):
            self._add_active_invitee()
        self.assertEqual(count_active_referrals(self.inviter), 4)
        granted = check_and_grant_referral_badges(self.inviter)
        self.assertEqual(granted, [])

    def test_grant_referral_5(self):
        for _ in range(5):
            self._add_active_invitee()
        self.assertEqual(count_active_referrals(self.inviter), 5)
        granted = check_and_grant_referral_badges(self.inviter)
        self.assertIn("referral_5", granted)

    def test_idempotency_for_referral_5(self):
        for _ in range(5):
            self._add_active_invitee()
        granted_1 = check_and_grant_referral_badges(self.inviter)
        granted_2 = check_and_grant_referral_badges(self.inviter)
        self.assertIn("referral_5", granted_1)
        self.assertEqual(granted_2, [])

    def test_grant_referral_10_after_5(self):
        for _ in range(5):
            self._add_active_invitee()
        g1 = check_and_grant_referral_badges(self.inviter)
        self.assertIn("referral_5", g1)

        for _ in range(5):
            self._add_active_invitee()
        self.assertEqual(count_active_referrals(self.inviter), 10)
        g2 = check_and_grant_referral_badges(self.inviter)
        self.assertIn("referral_10", g2)
