# apps/badges/management/commands/badges_top_roi.py

# stdlib
from collections import defaultdict
from decimal import Decimal, InvalidOperation

# django
from django.contrib.auth import get_user_model
from django.core.management.base import BaseCommand
from django.db.models import Sum

from apps.badges.models import Badge
from apps.badges.services import grant_badge

# local
from apps.stakes.models import Stake
from apps.token_app.models import Token
from apps.wallets.models import PendingReward

# پیکربندی
BADGE_KEY = "top_roi_s1"
BADGE_NAME = "Top ROI - Season 1"
BADGE_BP = 15  # 15bp = 0.15%
BADGE_IMAGE = "badge_top_roi_s1"
ROI_MIN = Decimal("1.50")  # حد آستانه، مثلا 1.5×

TOP_N = 1  # برترین(ها)؛ فعلاً 1 نفر


class Command(BaseCommand):
    help = "Compute ROI per user (USD-based) and grant 'top_roi_s1' to top-N users with ROI >= ROI_MIN."

    def handle(self, *args, **options):
        # 1) اطمینان از وجود Badge
        Badge.objects.update_or_create(
            key=BADGE_KEY,
            defaults={
                "name": BADGE_NAME,
                "reward_bonus_bp": BADGE_BP,
                "image_key": BADGE_IMAGE,
                "is_active": True,
            },
        )

        # 2) قیمت USD توکن‌ها
        token_price = dict(Token.objects.values_list("id", "usd_price"))

        # 3) مجموع Stake هر کاربر به USD
        #    sum over tokens: amount * token.usd_price
        staked_by_user = defaultdict(Decimal)
        stakes = Stake.objects.values("user_id", "token_id").annotate(total=Sum("amount"))
        for row in stakes:
            uid = row["user_id"]
            tok = row["token_id"]
            amt = row["total"] or Decimal("0")
            price = Decimal(str(token_price.get(tok, 0) or 0))
            staked_by_user[uid] += amt * price

        # 4) مجموع claimed rewards هر کاربر به USD
        claimed_by_user = defaultdict(Decimal)
        rewards = (
            PendingReward.objects.filter(status="claimed")  # اگر وضعیت دیگری هم ملاک است، اضافه کن
            .values("user_id", "token_id")
            .annotate(total=Sum("amount"))
        )
        for row in rewards:
            uid = row["user_id"]
            tok = row["token_id"]
            amt = row["total"] or Decimal("0")
            price = Decimal(str(token_price.get(tok, 0) or 0))
            claimed_by_user[uid] += amt * price

        # 5) ROI = claimed_usd / staked_usd  (اگر staked صفر، رد)
        roi_list = []
        user_ids = set(staked_by_user.keys()) | set(claimed_by_user.keys())
        for uid in user_ids:
            st_usd = staked_by_user.get(uid, Decimal("0"))
            cl_usd = claimed_by_user.get(uid, Decimal("0"))
            if st_usd > 0:
                try:
                    roi = cl_usd / st_usd
                except (InvalidOperation, ZeroDivisionError):
                    continue
                roi_list.append((uid, roi, st_usd, cl_usd))

        # 6) سورت نزولی بر اساس ROI
        roi_list.sort(key=lambda x: x[1], reverse=True)
        winners = [r for r in roi_list[:TOP_N] if r[1] >= ROI_MIN]

        # 7) اعطا Badge به برنده‌ها
        granted = 0
        U = get_user_model()
        for uid, roi, st_usd, cl_usd in winners:
            try:
                u = U.objects.get(id=uid)
            except U.DoesNotExist:
                continue
            if grant_badge(
                u,
                BADGE_KEY,
                meta={
                    "roi": str(roi.quantize(Decimal("0.0001"))),
                    "staked_usd": str(st_usd.quantize(Decimal("0.01"))),
                    "claimed_usd": str(cl_usd.quantize(Decimal("0.01"))),
                },
            ):
                granted += 1

        self.stdout.write(
            self.style.SUCCESS(
                f"Top ROI done. Evaluated {len(roi_list)} users; granted {granted} '{BADGE_KEY}' badges."
            )
        )
