# apps/badges/management/commands/badges_check_whales.py

# stdlib
from datetime import timedelta

# django
from django.contrib.auth import get_user_model
from django.core.management.base import BaseCommand
from django.db.models import Min
from django.utils import timezone

from apps.badges.models import Badge
from apps.badges.services import grant_badge

# local
from apps.stakes.models import Stake

# تنظیم‌پذیر: مثلاً 30 روز
WINDOW_DAYS = 30
BADGE_KEY = "whale_30d"
BADGE_NAME = "Whale - 30 Days"
BADGE_BP = 10  # 10bp = 0.10%
BADGE_IMAGE = "badge_whale_30"


class Command(BaseCommand):
    help = "Grant 'whale_30d' to users who kept an active stake for 30+ days (approximation by active stake age)."

    def handle(self, *args, **options):
        # 1) اطمینان از وجود Badge
        badge, _ = Badge.objects.update_or_create(
            key=BADGE_KEY,
            defaults={
                "name": BADGE_NAME,
                "reward_bonus_bp": BADGE_BP,
                "image_key": BADGE_IMAGE,
                "is_active": True,
            },
        )

        cutoff = timezone.now() - timedelta(days=WINDOW_DAYS)

        # 2) کاربرانی که حداقل یک Stake فعال دارند که createشان قدیمی‌تر از cutoff است
        # (MVP ساده: اگر یک استیک فعال 30+ روز داشته باشد → واجد شرایط)
        users_qs = (
            Stake.objects.filter(is_active=True)
            .values("user_id")
            .annotate(first_active=Min("created_at"))
            .filter(first_active__lte=cutoff)
        )

        granted = 0
        for row in users_qs:
            user_id = row["user_id"]

            U = get_user_model()
            try:
                u = U.objects.get(id=user_id)
            except U.DoesNotExist:
                continue
            if grant_badge(u, BADGE_KEY, meta={"window_days": WINDOW_DAYS}):
                granted += 1

        self.stdout.write(
            self.style.SUCCESS(f"Whale check done. Granted {granted} new '{BADGE_KEY}' badges.")
        )
