from django.contrib import admin

from .models import Ticket, TicketMessage, TicketStatus


class TicketMessageInline(admin.TabularInline):
    model = TicketMessage
    extra = 0
    readonly_fields = ["author", "from_staff", "created_at"]
    # فقط body رو عملاً ادیت می‌کنی، بقیه فقط نمایش هستن
    fields = ["author", "from_staff", "body", "created_at"]
    can_delete = False


@admin.register(Ticket)
class TicketAdmin(admin.ModelAdmin):
    list_display = ["id", "subject", "user", "category", "priority", "status", "updated_at"]
    list_filter = ["status", "category", "priority"]
    search_fields = ["subject", "user__username", "user__email"]
    inlines = [TicketMessageInline]
    readonly_fields = ["user", "created_at", "updated_at"]

    actions = ["make_closed"]

    @admin.action(description="Close selected tickets")
    def make_closed(self, request, queryset):
        queryset.update(status=TicketStatus.CLOSED)

    def save_formset(self, request, form, formset, change):
        """
        وقتی ادمین داخل تیکت پیام جدید اضافه می‌کند،
        اینجا author و from_staff را پر می‌کنیم تا دیگه author_id = NULL نشود.
        """
        instances = formset.save(commit=False)

        for obj in instances:
            # فقط روی TicketMessage‌ها کار داریم
            if isinstance(obj, TicketMessage):
                # پیام جدید (هنوز pk ندارد)
                if obj.pk is None:
                    # اگر به هر دلیل author ست نشده باشد، از یوزر فعلی ادمین استفاده کن
                    if obj.author_id is None:
                        obj.author = request.user
                        obj.from_staff = bool(request.user.is_staff)

            obj.save()

        # اگر m2m چیزی داشت
        formset.save_m2m()


@admin.register(TicketMessage)
class TicketMessageAdmin(admin.ModelAdmin):
    list_display = ["id", "ticket", "author", "from_staff", "created_at"]
    list_filter = ["from_staff", "created_at"]
    search_fields = ["ticket__subject", "author__username", "author__email", "body"]
