# stdlib
import bisect
from collections import defaultdict
from datetime import timedelta
from decimal import ROUND_DOWN, Decimal

# rest_framework
from rest_framework import status, viewsets
from rest_framework.exceptions import ValidationError
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response
from rest_framework.views import APIView
from rest_framework_simplejwt.authentication import JWTAuthentication

# django
from django.conf import settings
from django.db import IntegrityError, transaction
from django.db.models import Sum
from django.db.models.functions import TruncDate
from django.shortcuts import get_object_or_404
from django.utils import timezone

from apps.badges.services import user_badge_bonus_bp  # جمع bp بَج‌های کاربر
from apps.miners.models import Miner, UserMiner
from apps.plans.models import Plan

# local
from apps.plans.utils import build_absolute_image_url
from apps.reward.models import Reward
from apps.stakes.models import Stake
from apps.stakes.serializers import StakeSerializer
from apps.token_app.models import Token
from apps.users.services.referrals import compute_final_percent  # نرخ مؤثر با رفرال
from apps.users.services.referrals import count_active_referrals  # تعداد رفرال فعال
from apps.users.services.referrals import (
    get_current_referral_params,  # per_active_bonus & bonus_cap (Decimal)
)
from apps.wallets.models import WalletConnection
from django.db.models import Q

from apps.claims.models import ClaimTx
from apps.users.services.referrals import get_current_referral_params, count_active_referrals, compute_final_percent

from apps.wallets.service.balances import get_balances_for_user
from decimal import Decimal, InvalidOperation, ROUND_DOWN

import logging

logger = logging.getLogger(__name__)

SECONDS_IN_30D = Decimal("2592000")  # 30 روز


def _q_for_decimals(decimals):
    d = int(decimals or 0)
    d = max(0, min(d, 30))
    return Decimal(10) ** Decimal(-d)


def _base_percent_from_plan(miner) -> Decimal:
    plan = getattr(miner, "plan", None)
    val = getattr(plan, "monthly_reward_percent", None)
    return Decimal(str(val)) if val is not None else Decimal("4.5")


class StakeViewSet(viewsets.ModelViewSet):
    """
    CRUD برای Stake.
    قواعد ساخت (POST /stakes/):
      - برای هر کاربر روی هر توکن فقط یک Stake فعال مجاز است.
      - اگر قبلاً Stake فعال روی همان توکن وجود دارد، باید replace=true بدهی
        تا قبلی غیرفعال (و pause شود) و جدید ساخته شود.
      - در صورت ریس‌کانديشن، IntegrityError به پیام خوانا تبدیل می‌شود.
    """

    queryset = Stake.objects.all()
    serializer_class = StakeSerializer
    permission_classes = [IsAuthenticated]

    def get_queryset(self):
        uid = getattr(self.request.user, "id", None)
        if not uid:
            return Stake.objects.none()
        return Stake.objects.filter(user_id=uid).order_by("-created_at")

    def perform_create(self, serializer):
        # با توجه به create سفارشی، این عملاً صدا زده نمی‌شود،
        # اما اگر جایی از default create استفاده شد، فعال بماند.
        stake = serializer.save(user=self.request.user)
        if not getattr(stake, "is_active", False):
            stake.is_active = True
            stake.save(update_fields=["is_active"])

    def _resolve_token(self, request, vd):
        """
        تلاش برای استخراج Token از:
          1) validated_data['token']
          2) miner.token (اگر داشته باشیم)
          3) request.data['token'] به‌صورت id یا symbol (case-insensitive)
        """
        token = vd.get("token")
        miner = vd.get("miner")

        if not token and miner and getattr(miner, "token_id", None):
            token = miner.token

        if not token and "token" in request.data:
            raw = request.data.get("token")
            # اگر id داده شده (عددی)
            try:
                token_id = int(str(raw))
                t = Token.objects.filter(id=token_id).first()
                if t:
                    token = t
            except (TypeError, ValueError):
                # به عنوان symbol
                token_symbol = str(raw or "").strip()
                if token_symbol:
                    token = Token.objects.filter(symbol__iexact=token_symbol).first()

        return token

    def create(self, request, *args, **kwargs):
        # ولیدیشن ورودی (serializer خودش در create سخت‌گیر است)
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        user = request.user
        vd = serializer.validated_data

        token = self._resolve_token(request, vd)
        if not token:
            raise ValidationError("Token is required or could not be resolved.")

        # replace=true ?
        replace_raw = request.data.get("replace", False)
        replace = str(replace_raw).strip().lower() in ("1", "true", "yes", "on")

        # اگر replace=True → استیک فعال قبلی را غیرفعال و freeze کن
        if replace:
            now = timezone.now()
            Stake.objects.filter(user=user, token=token, is_active=True).update(
                is_active=False,
                paused_at=now,  # فریز برای محاسبات داشبورد
                auto_paused=False,  # این توقف به‌درخواست کاربر است، نه سیستم
                auto_pause_reason="",  # خالی یا متن دلخواه
            )
        else:
            # اگر replace=False و استیک فعال وجود دارد → خطا
            if Stake.objects.filter(user=user, token=token, is_active=True).exists():
                raise ValidationError(
                    "You already have an active stake for this token. Send replace=true to replace the active stake."
                )

        # ساخت امن + هندل IntegrityError
        try:
            with transaction.atomic():
                instance = serializer.save(user=user, token=token, is_active=True)

                # اگر مدل/سرویس‌هایت این متدها را دارند، ایمن صدا بزن:
                try:
                    # اولین بار last_calculated_at را ست کند (درصورت وجود چنین متدی)
                    if hasattr(instance, "touch_accumulator"):
                        instance.touch_accumulator()
                except Exception:
                    pass

                try:
                    from apps.stakes.services import accrue_until_now
                    # تسویه‌ی اولیه تا الان (بدون آرگومان now؛ چون امضا قبلاً ارور می‌داد)
                    accrue_until_now(instance)
                except Exception:
                    # اگر سرویس در دسترس نبود، ساخت استیک را خراب نکن
                    pass

        except IntegrityError as e:
            # نام constraint را مطابق دیتابیس خودت تنظیم کن
            msg = str(e)
            if "uniq_active_stake_per_user_token" in msg or "uniq_user_token_activeflag" in msg:
                raise ValidationError(
                    "You already have an active stake for this token. Send replace=true to replace the active stake."
                ) from e
            raise

        out = self.get_serializer(instance)
        return Response({"detail": "stake_created", "stake": out.data}, status=status.HTTP_201_CREATED)




def _first_non_none(*vals):
    for v in vals:
        if v:
            return v
    return None


class StakedMinerDashboardGetView(APIView):
    permission_classes = [IsAuthenticated]

    def get(self, request):
        try:
            stakes = (
                Stake.objects.filter(user=request.user)
                .select_related("miner", "token", "miner__plan")
                .order_by("-created_at")
            )
            if not stakes.exists():
                return Response({"detail": "No staked miners found."}, status=404)

            now = timezone.now()
            results = []
            total_power_effective = 0.0
            total_staked_all = Decimal(0)

            # ---- Referral/Badge bonuses (bp) ----
            ref_cfg = get_current_referral_params()
            per_active_bp = Decimal(ref_cfg["per_active_bonus"]) * Decimal(10000)
            cap_bp = Decimal(ref_cfg["bonus_cap"]) * Decimal(10000)
            active_refs = count_active_referrals(request.user)
            badge_bp = Decimal(user_badge_bonus_bp(request.user) or 0)

            ref_bp_est = min(Decimal(active_refs) * per_active_bp, cap_bp)
            total_bp_est = ref_bp_est + badge_bp

            # ---- Wallet live balances (optional) ----
            USE_LIVE = getattr(settings, "DASHBOARD_LIVE_BALANCE", False)
            wallet_balances = {}
            if USE_LIVE:
                try:
                    tokens = [s.token for s in stakes if getattr(s, "token", None)]
                    wallet_balances = get_balances_for_user(request.user, tokens=tokens)
                except Exception:
                    wallet_balances = {}

            # ---- Accrued per token (CLAMPED/FROZEN) ----
            accrued_active_by_symbol = defaultdict(Decimal)

            def _first_non_none(*vals):
                for v in vals:
                    if v:
                        return v
                return None

            for stake in stakes:
                miner = stake.miner
                token = stake.token
                sym = getattr(token, "symbol", None)
                decimals = int(getattr(token, "decimals", 9) or 9)
                q = _q_for_decimals(decimals)

                # amount
                try:
                    amount = Decimal(stake.amount)
                except (InvalidOperation, TypeError):
                    amount = Decimal("0")

                # rates
                base_percent = _base_percent_from_plan(miner)  # Decimal("x.xxxx")
                eff_percent = Decimal(str(compute_final_percent(request.user, base_percent)))
                rate_fraction = eff_percent / Decimal(100)
                earning_per_second = (amount * rate_fraction) / SECONDS_IN_30D  # Decimal

                # ---- state & balances
                auto_paused = bool(getattr(stake, "auto_paused", False))
                paused_at = getattr(stake, "paused_at", None)
                last_check = getattr(stake, "last_balance_check_at", None)

                stake_active = bool(getattr(stake, "is_active", False))
                miner_online = bool(getattr(miner, "is_online", False))

                wal_bal = wallet_balances.get(sym, None) if USE_LIVE else None
                required_for_this_stake = amount
                live_insufficient = False
                if wal_bal is not None:
                    try:
                        live_insufficient = wal_bal < required_for_this_stake
                    except Exception:
                        live_insufficient = False

                enough_balance = not live_insufficient
                effective_online = stake_active and miner_online and enough_balance and (not auto_paused)

                # ---- robust FREEZE point (بدون استفاده از updated_at)
                stake_deactivated_at = getattr(stake, "deactivated_at", None) or getattr(stake, "stopped_at", None)
                miner_offline_since = getattr(miner, "offline_since", None)
                miner_last_seen = getattr(miner, "last_seen_at", None) or getattr(miner, "last_online_at", None)

                freeze_at = None
                if auto_paused:
                    freeze_at = _first_non_none(paused_at, last_check)
                if not effective_online:
                    freeze_at = _first_non_none(
                        freeze_at,
                        stake_deactivated_at if not stake_active else None,
                        miner_offline_since if not miner_online else None,
                        miner_last_seen if not miner_online else None,
                        last_check,  # حداقل اگر آخرین چک داریم
                    )

                # NOTE: عمداً از updated_at استفاده نمی‌کنیم چون ممکن است دائماً تغییر کند
                # و باعث شود «فریز» عملاً حرکت کند.

                window_end = freeze_at or now

                started_at = getattr(stake, "created_at", None)
                if started_at is None:
                    seconds_active = Decimal(0)
                else:
                    sec = int((window_end - started_at).total_seconds())
                    seconds_active = Decimal(sec if sec > 0 else 0)

                accrued_for_window = (earning_per_second * seconds_active)

                # ---- status_reason
                if auto_paused:
                    status_reason = "insufficient_balance"
                else:
                    if not enough_balance:
                        status_reason = "insufficient_balance"
                    elif not miner_online:
                        status_reason = "miner_offline"
                    elif not stake_active:
                        status_reason = "stake_inactive"
                    else:
                        status_reason = "ok"

                # ---- UI numbers
                if effective_online:
                    earning_per_second_display = earning_per_second
                    accrued_reward_display = accrued_for_window
                    total_power_effective += float(miner.power or 0.0)
                else:
                    if freeze_at:
                        # فریز واقعی
                        earning_per_second_display = Decimal(0)
                        accrued_reward_display = accrued_for_window
                    else:
                        # اگر «نقطهٔ فریز» نداریم، به‌جای صفر، accrued را نشان می‌دهیم (UI بهتر)
                        # اما EPS=0 می‌ماند تا رشد لحظه‌ای روی UI دیده نشود
                        earning_per_second_display = Decimal(0)
                        accrued_reward_display = accrued_for_window

                # ---- Summary aggregation
                if sym and (effective_online or freeze_at or (not effective_online and accrued_for_window > 0)):
                    # اگر فریز نداریم ولی accrued داریم، آن را هم در summary لحاظ کن تا صفرِ نامفهوم نگیریم
                    accrued_active_by_symbol[sym] += accrued_for_window

                earning_per_second_q = earning_per_second_display.quantize(q, rounding=ROUND_DOWN)
                accrued_reward_q = accrued_reward_display.quantize(q, rounding=ROUND_DOWN)
                total_staked_all += amount

                plan = getattr(miner, "plan", None)
                raw_video = getattr(plan, "video_url", "") if plan else ""
                video_url = (
                    request.build_absolute_uri(raw_video)
                    if (raw_video and str(raw_video).startswith("/"))
                    else (raw_video or "")
                )

                results.append(
                    {
                        "stake_id": stake.id,
                        "miner_id": miner.id,
                        "miner_name": miner.name,
                        "is_online": effective_online,
                        "status_reason": status_reason,  # ok | insufficient_balance | miner_offline | stake_inactive
                        "auto_paused": auto_paused,
                        "plan_level": getattr(miner.plan, "level", None),
                        "power": miner.power,
                        "symbol": sym,
                        "staked_amount": str(amount),
                        "base_rate_percent": str(base_percent.quantize(Decimal("0.0001"))),
                        "bonus_breakdown": {
                            "referral_active_count": int(active_refs),
                            "referral_bonus_bp": int(ref_bp_est),
                            "badge_bonus_bp": int(badge_bp),
                            "total_bonus_bp": int(total_bp_est),
                            "cap_bp": int(cap_bp),
                        },
                        "earning_per_second": str(earning_per_second_q),
                        "active_since": stake.created_at.isoformat().replace("+00:00", "Z"),
                        "seconds_active": int(seconds_active),                 # اگر freeze_at داشته باشیم فریز می‌ماند
                        "accrued_reward_until_now": str(accrued_reward_q),     # در نبود freeze_at هم صفر نمی‌شود
                        "video_url": video_url,
                    }
                )

            # =============================
            #   Earnings Summary
            # =============================
            all_tokens = list(Token.objects.all())

            token_by_contract = {}
            for t in all_tokens:
                addr = getattr(t, "token_address", None) or getattr(t, "contract_address", None)
                if addr:
                    token_by_contract[str(addr).lower()] = t

            token_by_symbol = {}
            for t in all_tokens:
                sym = getattr(t, "symbol", None)
                if sym:
                    token_by_symbol[sym] = t

            withdrawn_by_symbol = defaultdict(Decimal)

            user_wallet = (
                WalletConnection.objects
                .filter(user=request.user)
                .values_list("wallet_address", flat=True)
                .first()
            )

            claim_filter = Q(user=request.user)
            if user_wallet:
                claim_filter |= Q(user_address__iexact=user_wallet)

            claim_qs = (
                ClaimTx.objects
                .filter(claim_filter)
                .values("token_address", "amount_scaled")
            )

            for row in claim_qs:
                caddr = (row.get("token_address") or "").lower()
                tok = token_by_contract.get(caddr)
                if not tok:
                    continue
                decs = int(getattr(tok, "decimals", 18) or 18)
                scale_q = Decimal(10) ** Decimal(-decs)
                scaled = Decimal(str(row.get("amount_scaled", 0)))
                human_amount = (scaled * scale_q)
                withdrawn_by_symbol[tok.symbol] += human_amount

            earnings_summary = []
            all_syms = set(accrued_active_by_symbol.keys()) | set(withdrawn_by_symbol.keys())
            for sym in sorted(all_syms):
                tok = token_by_symbol.get(sym)
                decs = int(getattr(tok, "decimals", 18) if tok else 18)
                q = _q_for_decimals(decs)

                accrued_total = accrued_active_by_symbol.get(sym, Decimal(0)).quantize(q, rounding=ROUND_DOWN)
                withdrawn = withdrawn_by_symbol.get(sym, Decimal(0)).quantize(q, rounding=ROUND_DOWN)
                pending = accrued_total - withdrawn
                if pending < 0:
                    pending = Decimal(0)
                pending = pending.quantize(q, rounding=ROUND_DOWN)

                earnings_summary.append(
                    {
                        "symbol": sym,
                        "total_accrued": str(accrued_total),
                        "total_withdrawn": str(withdrawn),
                        "pending": str(pending),
                    }
                )

            return Response(
                {
                    "total_power": total_power_effective,
                    "total_staked": str(total_staked_all),
                    "active_miners": sum(1 for m in results if m["is_online"]),
                    "miners": results,
                    "earnings_summary": earnings_summary,
                }
            )
        except Exception:
            logger.exception("staked-miner-dashboard-get failed")
            return Response({"detail": "server_error"}, status=500)

class StakedMinerDashboardPostView(APIView):
    permission_classes = [IsAuthenticated]

    def _abs_media_url(self, request, maybe_url: str | None) -> str:
        if not maybe_url:
            return ""
        if isinstance(maybe_url, str) and maybe_url.startswith("/"):
            return request.build_absolute_uri(maybe_url)
        return maybe_url

    def _get_image_url(self, miner, request):
        # miner.image → plan.image → ""
        try:
            img = getattr(miner, "image", None)
            if img and hasattr(img, "url"):
                return self._abs_media_url(request, img.url)
        except Exception:
            pass
        try:
            plan_img = getattr(miner.plan, "image", None)
            if plan_img and hasattr(plan_img, "url"):
                return self._abs_media_url(request, plan_img.url)
        except Exception:
            pass
        return ""

    def _get_video_url(self, miner, request):
        try:
            raw = getattr(getattr(miner, "plan", None), "video_url", None)
            return self._abs_media_url(request, raw)
        except Exception:
            return ""

    def _get_miner_for_level(self, level, token=None):
        qs = Miner.objects.filter(plan__level=level)
        if token is not None:
            qst = qs.filter(tokens=token)
            if qst.exists():
                return qst.first()
        return qs.first()

    def _build_three_levels(self, current_level):
        levels = list(Plan.objects.order_by("level").values_list("level", flat=True).distinct())
        if not levels:
            return []
        idx = bisect.bisect_left(levels, current_level)
        prev_idx = max(0, idx - 1)
        cur_idx = min(idx, len(levels) - 1)
        next_idx = min(idx + 1, len(levels) - 1)
        return [levels[prev_idx], levels[cur_idx], levels[next_idx]]

    def post(self, request):
        # 1) نیاز به اتصال کیف پول
        if not WalletConnection.objects.filter(user=request.user).exists():
            return Response(
                {"detail": "Please connect your wallet first."}, status=status.HTTP_400_BAD_REQUEST
            )

        # 2) جلوگیری از داشتن بیش از یک استیک فعال برای همان توکن
        token_symbol = (request.data.get("token") or "").strip()
        if not token_symbol:
            return Response({"detail": "token is required."}, status=status.HTTP_400_BAD_REQUEST)

        token = get_object_or_404(Token, symbol=token_symbol.upper())
        if Stake.objects.filter(user=request.user, token=token, is_active=True).exists():
            return Response(
                {"detail": "You already have an active miner for this token. Stop it first."},
                status=status.HTTP_400_BAD_REQUEST,
            )

        # 3) ورودی‌ها
        data = request.data
        user = request.user
        user_id = data.get("user_id")
        miner_id = data.get("miner_id")

        if user_id and int(user_id) != user.id and not user.is_staff:
            return Response({"detail": "Permission denied."}, status=403)

        if not user_id or not miner_id:
            return Response({"detail": "user_id and miner_id are required."}, status=400)

        target_user_id = int(user_id)

        # 4) لیست UserMiner‌های کاربر برای این ماینر
        stakes = UserMiner.objects.filter(user_id=target_user_id, miner_id=miner_id).select_related(
            "miner", "token", "miner__plan"
        )
        if not stakes.exists():
            return Response(
                {"detail": "No staked miner found for this user and miner."}, status=404
            )

        # مرجع
        userminer = stakes[0]
        current_miner = userminer.miner
        current_plan = current_miner.plan
        current_level = getattr(current_plan, "level", None)
        token = userminer.token or (
            current_miner.tokens.first() if current_miner.tokens.exists() else None
        )

        # 5) آمار کلی
        total_power = 0
        total_staked = 0
        miners_data = []

        for s in stakes:
            m = s.miner
            total_power += m.power
            total_staked += s.staked_amount
            miners_data.append(
                {
                    "id": m.id,
                    "tokens": [t.id for t in m.tokens.all()],
                    "name": m.name,
                    "staked_amount": str(s.staked_amount),
                    "power": m.power,
                    "is_online": s.is_online,  # وضعیت per-user
                    "created_at": m.created_at.isoformat().replace("+00:00", "Z"),
                    "user_id": target_user_id,
                    "image": self._get_image_url(m, request),
                    "video_url": self._get_video_url(m, request),  # ✅ اضافه شد
                }
            )

        # 6) سه پیشنهادِ لِول
        miner_suggestions = []
        if current_level is not None:
            for lvl in self._build_three_levels(current_level):
                m = self._get_miner_for_level(lvl, token=token) or self._get_miner_for_level(
                    lvl, token=None
                )
                if m:
                    miner_suggestions.append(
                        {
                            "id": m.id,
                            "name": m.name,
                            "image": self._get_image_url(m, request),
                            "video_url": self._get_video_url(m, request),
                            "active": (m.id == current_miner.id),
                            "plan_level": m.plan.level if m.plan else None,
                        }
                    )
                else:
                    miner_suggestions.append(
                        {
                            "id": None,
                            "name": None,
                            "image": "",
                            "video_url": "",
                            "active": False,
                            "plan_level": lvl,
                        }
                    )
        else:
            for m in Miner.objects.all()[:3]:
                miner_suggestions.append(
                    {
                        "id": m.id,
                        "name": m.name,
                        "image": self._get_image_url(m, request),
                        "video_url": self._get_video_url(m, request),
                        "active": (m.id == current_miner.id),
                        "plan_level": m.plan.level if m.plan else None,
                    }
                )
            while len(miner_suggestions) < 3:
                miner_suggestions.append(
                    {
                        "id": None,
                        "name": None,
                        "image": "",
                        "video_url": "",
                        "active": False,
                        "plan_level": None,
                    }
                )

        # 7) خروجی — بدون گراف، و با فیکس m → current_miner
        return Response(
            {
                "miner_power": total_power,
                "staked_tokens": total_staked,
                "total_miners": len(miners_data),
                "active_miners": sum(1 for m in miners_data if m["is_online"]),
                "miners": miners_data,
                "image": self._get_image_url(current_miner, request),  # ⬅️ فیکس: m نبود
                "video_url": self._get_video_url(current_miner, request),  # ⬅️ اضافه شد
                "miner_suggestions": miner_suggestions,
            }
        )
