# apps/stakes/tasks.py
from decimal import Decimal
from django.utils import timezone
from django.db import transaction
from celery import shared_task

from apps.stakes.models import Stake
from apps.wallets.service.balances import get_balances_for_user


@shared_task(bind=True, autoretry_for=(Exception,), retry_backoff=True, max_retries=3)
def auto_pause_insufficient_balance(self):
    now = timezone.now()
    active_qs = (
        Stake.objects.select_related("user", "token", "miner")
            .filter(is_active=True)
            .order_by("user_id")
    )

    current_user_id = None
    cache_balances = None
    cache_tokens = None

    for stake in active_qs.iterator():
        uid = stake.user_id
        if uid != current_user_id:
            tokens = [
                s.token for s in Stake.objects
                    .filter(user_id=uid, is_active=True)
                    .select_related("token")
            ]
            try:
                cache_balances = get_balances_for_user(stake.user, tokens=tokens)
            except Exception:
                cache_balances = {}
            current_user_id = uid

        sym = getattr(stake.token, "symbol", None)
        if not sym:
            continue

        wal_bal = cache_balances.get(sym, Decimal("0"))
        required = Decimal(stake.amount)

        if wal_bal < required:
            with transaction.atomic():
                st = Stake.objects.select_for_update().get(pk=stake.pk)
                if st.is_active:
                    st.is_active = False
                    st.auto_paused = True
                    st.paused_at = now
                    st.auto_pause_reason = "insufficient_balance"
                    st.last_balance_check_at = now
                    st.save(update_fields=[
                        "is_active", "auto_paused", "auto_pause_reason", "last_balance_check_at", "paused_at"
                    ])
        else:
            Stake.objects.filter(pk=stake.pk).update(last_balance_check_at=now)
