# apps/stakes/service.py

# stdlib
from __future__ import annotations

from dataclasses import dataclass
from datetime import timedelta
from decimal import ROUND_DOWN, Decimal
from hashlib import sha256

from django.db import transaction

# django
from django.utils import timezone

from apps.plans.models import Plan

# local
from apps.stakes.models import Stake
from apps.users.services.referrals import compute_final_percent  # خروجی: درصد ماهانه مانند 4.5
from apps.wallets.models import PendingReward

"""
Stake accrual services.

This module contains idempotent, time-based accrual logic for Stakes:
- accrue_until_now(stake): settles rewards up to 'now' and creates RewardCycle/PendingReward entries
- Helpers to compute effective percent per plan/user (delegates to users.services.referrals)

Rules:
- Never hold long DB locks; keep atomic sections short.
- Use Decimal math and quantize consistently.
"""

SECONDS_IN_MONTH = Decimal("2592000")  # 30 * 24 * 3600


def _effective_monthly_percent(user, stake: Stake) -> Decimal:
    """
    Return effective monthly percent for a (user, plan) pair.

    If plan has an explicit monthly_reward_percent use that as base,
    otherwise fall back to default (e.g., 4.5). Then apply referral/badge bonuses.
    """
    plan: Plan | None = getattr(getattr(stake, "miner", None), "plan", None)
    base = getattr(plan, "monthly_reward_percent", None)
    if base is None:
        base = Decimal("4.5")
    return Decimal(str(compute_final_percent(user, Decimal(str(base)))))


def _rate_hash(user, stake: Stake) -> str:
    """
    هش نرخ فعلی برای تشخیص تغییر نرخ در طول زمان.
    فعلاً فقط plan% و referral% را لحاظ می‌کنیم. بعداً badgeها را اضافه می‌کنیم.
    """
    plan = getattr(getattr(stake, "miner", None), "plan", None)
    plan_pct = getattr(plan, "monthly_reward_percent", Decimal("4.5"))
    # bonus از Profile (inside compute_final_percent) به‌صورت ضمنی هست؛
    # برای هش، فقط ورودی‌های خام را می‌آوریم:
    raw = f"{plan_pct}|user:{user.id}"
    return sha256(raw.encode()).hexdigest()


@dataclass
class AccrualResult:
    added: Decimal
    total_pending: Decimal
    new_hash: str


ACCRUAL_RESOLUTION = timedelta(minutes=1)


@transaction.atomic
def accrue_until_now(stake):
    """
    Settle stake rewards up to *cutoff* time, where cutoff = min(now, paused_at).
    If stake is paused/offline/inactive, no earning is created; we only advance last_accrual_at
    to the cutoff so we won't accrue this window again later.
    """
    now = timezone.now()

    # نقطه شروع بازهٔ جدید برای تسویه
    last = getattr(stake, "last_accrual_at", None) or stake.created_at

    # اگر stake تا جایی متوقف شده، کات‌آف را همان بگذار؛ وگرنه تا الآن
    paused_at = getattr(stake, "paused_at", None)
    cutoff = min(now, paused_at) if paused_at else now

    # اگر بازه‌ای برای تسویه وجود ندارد، فقط نشانگر را جلو ببر (به‌خصوص در pause)
    if cutoff <= last:
        if cutoff != getattr(stake, "last_accrual_at", None):
            stake.last_accrual_at = cutoff
            stake.save(update_fields=["last_accrual_at"])
        return Decimal("0")

    # ریزولوشن؛ اگر خیلی کوتاه است، فقط نشانگر را جلو ببریم
    if cutoff <= last + ACCRUAL_RESOLUTION:
        stake.last_accrual_at = cutoff
        stake.save(update_fields=["last_accrual_at"])
        return Decimal("0")

    # شرط‌های کسب سود: اگر غیرفعال/متوقف/آفلاین است → هیچ سودی نده، فقط last را تا cutoff جلو ببر
    is_active = bool(getattr(stake, "is_active", True))
    auto_paused = bool(getattr(stake, "auto_paused", False))
    if (not is_active) or auto_paused:
        stake.last_accrual_at = cutoff
        stake.save(update_fields=["last_accrual_at"])
        return Decimal("0")

    # محاسبه نرخ
    plan = getattr(getattr(stake, "miner", None), "plan", None)
    base_percent = getattr(plan, "monthly_reward_percent", None) or Decimal("4.5")
    try:
        eff_percent = compute_final_percent(stake.user, base_percent)
    except Exception:
        eff_percent = Decimal(base_percent)

    # نرخ ساعتی از درصد ماهانه (۳۰ روز ~ ۷۲۰ ساعت)
    hourly_rate = (Decimal(eff_percent) / Decimal("100")) / Decimal("720")

    # مقدار بازه
    elapsed = cutoff - last
    hours = Decimal(str(elapsed.total_seconds())) / Decimal("3600")

    # amount * hourly_rate * hours
    amount = Decimal(str(stake.amount))
    earn = (amount * hourly_rate * hours).quantize(Decimal("0.00000001"), rounding=ROUND_DOWN)

    # صرف‌نظر از صفر/منفی (اگر ریزولوشن باعث شد)، فقط نشانگر را جلو ببریم
    stake.last_accrual_at = cutoff
    stake.save(update_fields=["last_accrual_at"])

    if earn <= 0:
        return Decimal("0")

    # ثبت درآمد (فقط تا cutoff)
    PendingReward.objects.create(
        user=stake.user,
        token=stake.token,
        amount=earn,
        status="pending",
        # اگر فیلدهای مرجع دوره/زمان دارید، اینجا ست کنید (مثلاً settled_until=cutoff)
    )
    return earn


def stake_is_earning(stake, wal_bal=None, required=None):
    """حق کسب سود دارد؟"""
    if not getattr(stake, "is_active", True):
        return False
    if getattr(stake, "auto_paused", False):
        return False

    # اگر موجودی را هم داری:
    if wal_bal is not None and required is not None:
        if wal_bal < required:
            return False
    return True


def earning_window(stake, now=None):
    """بازهٔ زمانی مجاز برای محاسبه: از آخرین مبنا تا min(now, paused_at)."""
    now = now or timezone.now()
    # نقطه شروع را با منطق خودت ست کن:
    start = getattr(stake, "last_reward_at", None) or getattr(stake, "active_since", None) or getattr(stake,"created_at",None)
    end = getattr(stake, "paused_at", None) or now
    if start and end and end > start:
        return start, end
    return None, None
