from decimal import Decimal, InvalidOperation
from rest_framework import serializers
from apps.stakes.models import Stake
from apps.wallets.models import WalletConnection
from apps.wallets.service.balances import get_balances_for_user  # همونی که قبلا استفاده می‌کردی

class StakeSerializer(serializers.ModelSerializer):
    class Meta:
        model = Stake
        fields = [
            "id",
            "user",
            "miner",
            "amount",
            "created_at",
            "token",
            "is_active",
        ]
        read_only_fields = ["id", "created_at", "user"]  # user همیشه از request می‌آید

    # --- ولیدیشن کمکی فقط برای ساخت ---
    def _validate_on_create(self, data):
        request = self.context.get("request")
        if not request or not getattr(request, "user", None) or not request.user.is_authenticated:
            raise serializers.ValidationError("Authentication required.")
        user = request.user

        miner = data.get("miner")
        amount = data.get("amount")
        token = data.get("token")

        # الزامی بودن فیلدهای پایه
        if not miner:
            raise serializers.ValidationError({"miner": ["Miner is required."]})
        if not token:
            raise serializers.ValidationError({"token": ["Token is required for staking."]})
        if amount is None:
            raise serializers.ValidationError({"amount": ["Amount is required."]})

        # پارس amount
        try:
            amt = Decimal(str(amount))
        except (InvalidOperation, TypeError):
            raise serializers.ValidationError({"amount": ["Invalid amount."]})
        if amt <= 0:
            raise serializers.ValidationError({"amount": ["Amount must be greater than zero."]})

        # حداقل پلن
        plan_price = getattr(getattr(miner, "plan", None), "price", None)
        if plan_price is not None:
            try:
                min_req = Decimal(str(plan_price))
                if amt < min_req:
                    sym = getattr(token, "symbol", "") or ""
                    raise serializers.ValidationError(
                        {"amount": [f"Amount must be at least {min_req} {sym} for this plan."]}
                    )
            except (InvalidOperation, TypeError):
                pass  # اگر price بد بود، از این چک رد می‌شویم

        # اتصال ولت
        if not WalletConnection.objects.filter(user=user).exists():
            raise serializers.ValidationError("Please connect your wallet first.")

        # موجودی ولت برای همان توکن
        try:
            balances = get_balances_for_user(user, tokens=[token])  # {"SYM": Decimal(...)}
        except Exception:
            raise serializers.ValidationError(
                "Unable to read wallet balance. Please try again shortly."
            )

        sym = getattr(token, "symbol", None)
        if not sym:
            raise serializers.ValidationError("Token symbol is missing.")

        wallet_bal = balances.get(sym)
        if wallet_bal is None:
            raise serializers.ValidationError({ "token": [f"Wallet balance for {sym} not available."] })

        if amt > wallet_bal:
            raise serializers.ValidationError({
                "amount": [f"Insufficient {sym} balance. Have {wallet_bal}, trying to stake {amt}."]
            })

        return data

    def validate(self, data):
        """
        اگر self.instance = None یعنی create
        اگر self.instance != None یعنی update/PATCH
        """
        if self.instance is None:
            # CREATE
            return self._validate_on_create(data)
        else:
            # UPDATE/PATCH: فقط فیلدهای ارسالی را چک کن.
            # جلوگیری از تغییر فیلدهای پایه در آپدیت
            forbidden = []
            for f in ("token", "miner", "amount", "user"):
                if f in data:
                    forbidden.append(f)
            if forbidden:
                raise serializers.ValidationError(
                    {f: ["This field cannot be updated."] for f in forbidden}
                )

            # اگر فقط is_active آمده، اجازه بده (چک اضافی لازم نیست)
            return data

    def create(self, validated_data):
        request = self.context.get("request")
        if not request or not hasattr(request, "user") or request.user.is_anonymous:
            raise serializers.ValidationError("Authentication required.")

        validated_data["user"] = request.user
        obj = Stake.objects.create(**validated_data)

        # اگر به هر دلیلی is_active در مدل False است، هنگام ساخت فعالش کنیم
        if hasattr(obj, "is_active") and not obj.is_active:
            obj.is_active = True
            obj.save(update_fields=["is_active"])
        return obj

    def update(self, instance, validated_data):
        # تشخیص تغییر is_active
        is_active_changed = False
        if "is_active" in validated_data:
            old = getattr(instance, "is_active", None)
            new = validated_data.get("is_active")
            if (old is not None) and (new is not None) and (old != new):
                is_active_changed = True

        # اعمال فقط فیلدهای مجاز (الان عملاً فقط is_active)
        for k, v in validated_data.items():
            setattr(instance, k, v)
        instance.save()

        # آپدیت فلگ رفرال کاربر بعد از تغییر وضعیت
        if is_active_changed:
            try:
                from apps.users.services.referrals import set_referral_active_flag
                has_active = type(instance).objects.filter(user=instance.user, is_active=True).exists()
                set_referral_active_flag(instance.user, has_active)
            except Exception:
                pass

        return instance