from drf_yasg.utils import swagger_auto_schema
from rest_framework import status, viewsets
from rest_framework.decorators import action, api_view, permission_classes
from rest_framework.permissions import AllowAny, IsAuthenticated
from rest_framework.response import Response
from rest_framework.throttling import ScopedRateThrottle
from rest_framework.views import APIView

from django.core.cache import cache
from django.db.models import Prefetch

from apps.plans.models import Plan
from apps.plans.utils import build_absolute_image_url
from apps.token_app.models import Token

from .models import Miner, UserMiner
from .serializers import MinerLightSerializer, MinerPublicSerializer, MinerSerializer


class MinerViewSet(viewsets.ModelViewSet):
    queryset = Miner.objects.all()
    serializer_class = MinerSerializer

    def perform_create(self, serializer):
        plan = serializer.validated_data.get("plan")
        power = plan.power if plan else 0
        serializer.save(power=power)

    def perform_update(self, serializer):
        plan = serializer.validated_data.get("plan")
        power = plan.power if plan else serializer.instance.power
        serializer.save(power=power)

    @action(detail=False, methods=["get"], url_path="me", permission_classes=[IsAuthenticated])
    def my_miners(self, request):
        user = request.user
        user_miners = UserMiner.objects.filter(user=user).select_related("miner", "token")
        data = []
        for um in user_miners:
            miner = um.miner
            data.append(
                {
                    "id": miner.id,
                    "tokens": [t.id for t in miner.tokens.all()],
                    "name": miner.name,
                    "staked_amount": str(um.staked_amount),
                    "power": miner.power,
                    "user_power": getattr(um, "power", miner.power),
                    "is_online": um.is_online,
                    "created_at": miner.created_at.isoformat() + "Z",
                    "image": build_absolute_image_url(request, getattr(miner.plan, "image", None)),
                }
            )
        return Response(data, status=status.HTTP_200_OK)


class MinerDetailView(APIView):
    permission_classes = [IsAuthenticated]

    def get(self, request, miner_id):
        user = request.user
        try:
            if user.is_staff:
                miner = Miner.objects.get(id=miner_id)
            else:
                userminer = UserMiner.objects.get(user=user, miner_id=miner_id)
                miner = userminer.miner
        except Miner.DoesNotExist:
            return Response(
                {"detail": "You do not have permission to access this miner."}, status=403
            )
        except UserMiner.DoesNotExist:
            return Response(
                {"detail": "You do not have permission to access this miner."}, status=403
            )

        serializer = MinerSerializer(miner, context={"request": request})
        return Response(serializer.data, status=200)


class PublicCatalogMetricsView(APIView):
    permission_classes = [AllowAny]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "public"

    @swagger_auto_schema(
        operation_summary="Public catalog metrics",
        operation_description="Lightweight metrics: totals of plans, miners (total/online), distinct tokens used by miners.",
        responses={200: "OK"},
    )
    def get(self, request, *args, **kwargs):
        # plans
        from apps.plans.models import Plan

        plans_total = Plan.objects.count()

        # miners
        miners_total = Miner.objects.count()
        miners_online = Miner.objects.filter(is_online=True).count()

        # distinct tokens used by miners
        token_ids = (
            Miner.objects.values_list("tokens__id", flat=True).distinct().exclude(tokens__id=None)
        )
        distinct_token_count = len([tid for tid in token_ids if tid is not None])

        return Response(
            {
                "plans_total": plans_total,
                "miners_total": miners_total,
                "miners_online": miners_online,
                "distinct_tokens_in_miners": distinct_token_count,
            },
            status=status.HTTP_200_OK,
        )


@api_view(["POST"])
@permission_classes([IsAuthenticated])
def get_miner_by_amount(request):
    try:
        amount = float(request.data.get("amount"))
        token_id = int(request.data.get("token"))  # کلاینت فیلد را "token" می‌فرستد
    except (TypeError, ValueError):
        return Response({"detail": "Invalid amount or token"}, status=400)

    if amount <= 0:
        return Response({"detail": "Amount must be greater than 0"}, status=400)

    try:
        token = Token.objects.get(id=token_id)
    except Token.DoesNotExist:
        return Response({"detail": "Token not found"}, status=404)

    plan = Plan.objects.filter(price__lte=amount, tokens=token).order_by("-price").first()
    if not plan:
        return Response({"detail": "No suitable plan found"}, status=404)

    miner = Miner.objects.filter(plan=plan, tokens=token).first()
    if not miner:
        return Response({"detail": "No miner found for selected plan and token."}, status=404)

    data = {
        "id": miner.id,
        "tokens": [t.id for t in miner.tokens.all()],
        "name": miner.name,
        "staked_amount": str(miner.staked_amount),
        "power": miner.power,
        "is_online": miner.is_online,
        "created_at": miner.created_at.isoformat() + "Z",
        "image": build_absolute_image_url(request, getattr(miner.plan, "image", None)),
    }
    return Response(data, status=200)


class MinerListPublicView(APIView):
    """
    GET /api/miners/public/?token=RZ&online=true&limit=3&offset=0&light=true
    """

    permission_classes = [AllowAny]
    throttle_classes = [ScopedRateThrottle]
    throttle_scope = "public"

    CACHE_TTL = 30  # seconds

    def _build_cache_key(self, token, plan_id, online, limit, offset, light):
        return f"miners_public:tk={token or '_'}:pl={plan_id or '_'}:on={online or '_'}:l={limit}:o={offset}:light={1 if light else 0}"

    def _get_queryset_filtered(self, token_symbol, plan_id, online):
        qs = (
            Miner.objects.select_related("plan")
            .prefetch_related(Prefetch("tokens", queryset=Token.objects.only("id", "symbol")))
            .all()
            .order_by("-created_at")
        )
        if token_symbol:
            qs = qs.filter(tokens__symbol__iexact=token_symbol)
        if plan_id and str(plan_id).isdigit():
            qs = qs.filter(plan_id=int(plan_id))
        if online in ("true", "1", "yes"):
            qs = qs.filter(is_online=True)
        elif online in ("false", "0", "no"):
            qs = qs.filter(is_online=False)
        return qs

    def get(self, request, *args, **kwargs):
        token_symbol = (request.query_params.get("token") or "").strip()
        plan_id = request.query_params.get("plan")
        online = (request.query_params.get("online") or "").strip().lower()
        light = (request.query_params.get("light") or "").strip().lower() in ("1", "true", "yes")

        try:
            limit = min(max(int(request.query_params.get("limit", 20)), 1), 100)
        except Exception:
            limit = 20
        try:
            offset = max(int(request.query_params.get("offset", 0)), 0)
        except Exception:
            offset = 0

        cache_key = self._build_cache_key(token_symbol, plan_id, online, limit, offset, light)
        cached = cache.get(cache_key)
        if cached is not None:
            return Response(cached, status=200)

        qs = self._get_queryset_filtered(token_symbol, plan_id, online)
        total = qs.count()
        items = qs[offset : offset + limit]

        if light:
            ser = MinerLightSerializer(items, many=True)
        else:
            ser = MinerPublicSerializer(items, many=True)

        payload = {"count": total, "limit": limit, "offset": offset, "results": ser.data}

        try:
            cache.set(cache_key, payload, timeout=self.CACHE_TTL)
        except Exception:
            pass

        return Response(payload, status=200)
