from decimal import Decimal

from rest_framework.decorators import api_view, permission_classes
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response
from django.utils import timezone
from .services import update_miners_active_window_progress
from apps.users.services.referrals import compute_final_percent
from apps.events.models import Task, UserEnergyBoost, UserTemporaryMinerAccess
from apps.events.serializers import EnergyBoostSerializer, MinerAccessSerializer
from apps.events.serializers_dashboard import TaskWithProgressSerializer
from apps.events.services import (
    register_login_activity,
    update_login_streak_progress,
    ensure_login_activity_for_today,
    expire_time_limited_rewards,
    evaluate_and_reward_for_user
)


@api_view(["GET"])
# @permission_classes([IsAuthenticated])
def events_dashboard_view(request):
    user = request.user
    # ensure_login_activity_for_today(user)
    evaluate_and_reward_for_user(user)
    expire_time_limited_rewards()
    update_miners_active_window_progress(user)
    now = timezone.now()
    # Count this request as today's login activity for streak tasks
    # update_login_streak_progress(user)

    tasks_qs = Task.objects.filter(is_active=True)
    active_boosts_qs = UserEnergyBoost.objects.filter(
        user=user, is_active=True, ends_at__gte=now
    )
    active_miner_accesses_qs = UserTemporaryMinerAccess.objects.filter(
        user=user, is_active=True, ends_at__gte=now
    )

    # درصد فعلی با رفرنس
    base_percent = Decimal("1.1")
    current_percent = compute_final_percent(user, base_percent, at=now)

    # 🔹 اینجا مستقیم از serializerهای جزئی استفاده می‌کنیم
    tasks_data = TaskWithProgressSerializer(
        tasks_qs, many=True, context={"request": request}
    ).data
    boosts_data = EnergyBoostSerializer(active_boosts_qs, many=True).data
    miners_data = MinerAccessSerializer(active_miner_accesses_qs, many=True).data

    # اگر می‌ترسی Decimal مشکل بده، می‌تونی به float/str تبدیلش کنی:
    current_percent_out = float(current_percent)

    return Response(
        {
            "tasks": tasks_data,
            "active_boosts": boosts_data,
            "active_miner_accesses": miners_data,
            "now": now,
            "current_percent": current_percent_out,
        },
        status=200,
    )
