from rest_framework import serializers
from apps.miners.models import UserMiner
from apps.stakes.models import Stake
from .models import (
    EnergyTurboDefinition,
    MinerAccessDefinition,
    Task,
    UserEnergyBoost,
    UserTemporaryMinerAccess,
)


class TaskSerializer(serializers.ModelSerializer):
    reward_def_name = serializers.SerializerMethodField()
    image = serializers.ImageField(read_only=True)

    class Meta:
        model = Task
        fields = [
            "id",
            "slug",
            "title",
            "description",
            "kind",
            "condition_code",
            "reward_type",
            "reward_def_name",
            "image",  # URL کامل تصویر
        ]

    def get_reward_def_name(self, obj):
        try:
            return str(obj.reward_object)
        except Exception:
            return None


class EnergyBoostSerializer(serializers.ModelSerializer):
    definition_name = serializers.CharField(source="definition.name", read_only=True)
    definition_image = serializers.ImageField(source="definition.image", read_only=True)

    class Meta:
        model = UserEnergyBoost
        fields = [
            "id",
            "definition",
            "definition_name",
            "definition_image",
            "miner",
            "starts_at",
            "ends_at",
            "is_active",
        ]



class MinerAccessSerializer(serializers.ModelSerializer):
    definition_name = serializers.CharField(source="definition.name", read_only=True)
    definition_image = serializers.ImageField(source="definition.image", read_only=True)

    class Meta:
        model = UserTemporaryMinerAccess
        fields = [
            "id",
            "definition",
            "definition_name",
            "definition_image",
            "starts_at",
            "ends_at",
            "is_active",
        ]

class EnergyTurboDefinitionSerializer(serializers.ModelSerializer):
    """تعریف کلی Turboها (برای ادمین تعریف شده)"""

    class Meta:
        model = EnergyTurboDefinition
        fields = [
            "id",
            "name",
            "percent_boost",
            "duration_seconds",
            "image",
            "is_active",
        ]


class MinerAccessDefinitionSerializer(serializers.ModelSerializer):
    """تعریف کلی Premium Miner Access"""

    miner_name = serializers.CharField(source="miner.name", read_only=True)

    class Meta:
        model = MinerAccessDefinition
        fields = [
            "id",
            "name",
            "miner",        # id ماینر
            "miner_name",   # اسم ماینر
            "duration_seconds",
            "image",
            "is_active",
        ]


class EventsCatalogSerializer(serializers.Serializer):
    """
    کاتالوگ کلی ایونت‌ها برای فرانت:
    - تسک‌های فعال
    - Energy Turbo definition های فعال
    - Miner Access definition های فعال
    """

    tasks = TaskSerializer(many=True)
    energy_turbos = EnergyTurboDefinitionSerializer(many=True)
    miner_accesses = MinerAccessDefinitionSerializer(many=True)




class RestakeInitiateSerializer(serializers.Serializer):
    stake_id = serializers.IntegerField()
    symbol = serializers.CharField(max_length=32)

    def validate(self, attrs):
        request = self.context["request"]
        user = request.user

        stake_id = attrs["stake_id"]
        symbol = attrs["symbol"]

        try:
            # stake باید مال همین یوزر باشد و active (اگر فیلد is_active داری هم می‌توانی چک کنی)
            stake = (
                Stake.objects
                .select_related("miner")
                .get(id=stake_id, user=user)
            )
        except Stake.DoesNotExist:
            raise serializers.ValidationError({"stake_id": "Invalid or inactive stake"})

        # اگر خواستی می‌تونی سمبل رو با توکن stake هم چک کنی:
        # if stake.token and stake.token.symbol.lower() != symbol.lower():
        #     raise serializers.ValidationError({"symbol": "Symbol does not match stake token"})

        # برای بقیه‌ی منطق، همچنان miner لازم داریم
        attrs["stake"] = stake
        attrs["miner"] = stake.miner  # این همون UserMiner است
        return attrs