# apps/claims/models.py
from django.conf import settings
from django.db import models
from django.contrib.postgres.fields import JSONField  # or models.JSONField in newer Django

from apps.miners.models import UserMiner  # adjust import to your project


class ClaimPurpose(models.TextChoices):
    WITHDRAW = "withdraw", "Withdraw to wallet"
    RESTAKE = "restake", "Restake rewards"


class ClaimIntent(models.Model):
    STATUS_CHOICES = [
        ("pending", "pending"),
        ("confirmed", "confirmed"),
        ("expired", "expired"),
        ("cancelled", "cancelled"),
    ]

    user = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.CASCADE)
    token_address = models.CharField(max_length=64, db_index=True)
    token_symbol = models.CharField(max_length=16, db_index=True)
    user_address = models.CharField(max_length=64, db_index=True)
    amount_scaled = models.DecimalField(max_digits=65, decimal_places=0)
    nonce = models.BigIntegerField()
    deadline = models.BigIntegerField()
    signature = models.TextField()  # بهتر از BinaryField برای JSON/REST
    status = models.CharField(max_length=16, choices=STATUS_CHOICES, default="pending")
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    purpose = models.CharField(
        max_length=20,
        choices=ClaimPurpose.choices,
        default=ClaimPurpose.WITHDRAW,
    )

    target_miner = models.ForeignKey(
        UserMiner,
        null=True,
        blank=True,
        on_delete=models.PROTECT,
        related_name="restake_claims",
        help_text="Target miner for restake operations",
    )

    meta = models.JSONField(default=dict, blank=True)

    class Meta:
        unique_together = [("user", "nonce")]
        indexes = [
            models.Index(fields=["user_address"]),
            models.Index(fields=["nonce"]),
        ]


class ClaimTx(models.Model):
    user = models.ForeignKey(
        settings.AUTH_USER_MODEL, on_delete=models.SET_NULL, null=True, blank=True
    )
    user_address = models.CharField(max_length=64, db_index=True)
    token_address = models.CharField(max_length=64, db_index=True)
    token_symbol = models.CharField(max_length=16, db_index=True)
    amount_scaled = models.DecimalField(max_digits=65, decimal_places=0)
    nonce = models.BigIntegerField()
    tx_hash = models.CharField(max_length=80, unique=True)
    created_at = models.DateTimeField(auto_now_add=True)

    class Meta:
        indexes = [
            models.Index(fields=["user_address", "token_address"]),
            models.Index(fields=["nonce"]),
        ]
