
# apps/badges/services.py
from __future__ import annotations

from collections import defaultdict
from datetime import timedelta
from decimal import Decimal, InvalidOperation

from django.contrib.auth import get_user_model
from django.db import transaction
from django.db.models import Sum, Q
from django.utils import timezone

from apps.badges.models import Badge, UserBadge
from apps.stakes.models import Stake
from apps.token_app.models import Token

try:
    from apps.users.services.referrals import count_active_referrals
except Exception:
    count_active_referrals = None

User = get_user_model()

def _safe_getattr(obj, attr, default=None):
    try:
        return getattr(obj, attr)
    except Exception:
        return default

def _bp_to_decimal(bp: int) -> Decimal:
    return (Decimal(int(bp)) / Decimal(10000))

def _decimal(val) -> Decimal:
    try:
        return Decimal(str(val or "0"))
    except InvalidOperation:
        return Decimal("0")

def user_badge_bonus_bp(user) -> int:
    """Sum of reward_rate_bp for user's ACTIVE badges."""
    qs = UserBadge.objects.select_related("badge").filter(user=user)
    if "is_active" in [f.name for f in UserBadge._meta.get_fields()]:
        qs = qs.filter(is_active=True)
    total = qs.aggregate(total=Sum("badge__reward_rate_bp")).get("total") or 0
    try:
        return int(total)
    except Exception:
        return int(_decimal(total))

def grant_badge(user: User, badge: Badge | str) -> bool:
    """Ensure a (user, badge) pair exists and is active. Returns True if created/activated."""
    if isinstance(badge, str):
        badge = Badge.objects.filter(Q(key=badge) | Q(name=badge) | Q(title=badge)).first()
        if not badge:
            return False
    with transaction.atomic():
        ub, created = UserBadge.objects.select_for_update().get_or_create(user=user, badge=badge)
        changed = False
        if hasattr(ub, "is_active") and not ub.is_active:
            ub.is_active = True
            ub.save(update_fields=["is_active"])
            changed = True
        return created or changed

def revoke_badge(user: User, badge: Badge | str) -> bool:
    if isinstance(badge, str):
        badge = Badge.objects.filter(Q(key=badge) | Q(name=badge) | Q(title=badge)).first()
        if not badge:
            return False
    try:
        ub = UserBadge.objects.get(user=user, badge=badge)
    except UserBadge.DoesNotExist:
        return False
    if not hasattr(ub, "is_active") or ub.is_active:
        if hasattr(ub, "is_active"):
            ub.is_active = False
            ub.save(update_fields=["is_active"])
        else:
            # if no is_active flag, delete
            ub.delete()
        return True
    return False

# -------- Rules --------

def _first_n_users_award():
    for b in Badge.objects.filter(is_active=True, type="FIRST_N").exclude(limit_count__isnull=True):
        n = int(b.limit_count or 0)
        if n <= 0:
            continue
        order_field = "date_joined" if hasattr(User, "date_joined") else "id"
        early_users = User.objects.all().order_by(order_field).values_list("id", flat=True)[:n]
        for uid in early_users:
            u = User.objects.filter(id=uid).first()
            if not u:
                continue
            grant_badge(u, b)

def _referral_thresholds_award_for_user(user: User):
    ref_badges = Badge.objects.filter(is_active=True, type="REFERRAL").exclude(referral_threshold__isnull=True)
    if not ref_badges.exists():
        return
    if count_active_referrals:
        rc = int(count_active_referrals(user))
    else:
        try:
            from apps.users.models import ReferralRelation
            rc = ReferralRelation.objects.filter(inviter=user, active=True).count()
        except Exception:
            rc = 0
    for b in ref_badges:
        thr = int(b.referral_threshold or 0)
        if thr and rc >= thr:
            grant_badge(user, b)

def _level5_starter_award_for_user(user: User):
    try:
        from apps.miners.models import Miner
        from apps.plans.models import Plan
    except Exception:
        return
    qs = Stake.objects.filter(user=user, is_active=True).select_related("miner__plan")
    for s in qs:
        plan = _safe_getattr(_safe_getattr(s, "miner", None), "plan", None)
        lvl = _safe_getattr(plan, "level", 0)
        if int(lvl or 0) >= 5:
            b = Badge.objects.filter(is_active=True, type="LEVEL5").first()
            if b:
                grant_badge(user, b)
            break

def _loyal_whale_award_for_user(user: User):
    now = timezone.now()
    for b in Badge.objects.filter(is_active=True, type="WHALE").exclude(whale_months__isnull=True):
        months = int(b.whale_months or 0)
        if months <= 0:
            continue
        since = now - timedelta(days=30 * months)
        eligible = Stake.objects.filter(user=user, is_active=True, created_at__lte=since).exists()
        if eligible:
            grant_badge(user, b)
        else:
            revoke_badge(user, b)

def _compute_principal_usd_for_user(user: User) -> Decimal:
    total = Decimal("0")
    stakes = Stake.objects.filter(user=user).values("token_id").annotate(total=Sum("amount"))
    token_prices = {t.id: _decimal(t.usd_price) for t in Token.objects.all()}
    for row in stakes:
        amt = _decimal(row["total"] or 0)
        price = token_prices.get(row["token_id"], Decimal("0"))
        total += (amt * price)
    return total

def _compute_rewards_claimed_usd_for_user(user: User) -> Decimal:
    try:
        from apps.wallets.models import PendingReward
    except Exception:
        return Decimal("0")
    qs = PendingReward.objects.filter(user=user, withdrawn=True).select_related("token")
    total = Decimal("0")
    for r in qs:
        price = _decimal(_safe_getattr(r.token, "usd_price", 0))
        total += _decimal(r.amount) * price
    return total

def compute_user_roi_multiple(user: User, include_pending: bool = True) -> Decimal:
    principal = _compute_principal_usd_for_user(user)
    if principal <= 0:
        return Decimal("0")
    claimed = _compute_rewards_claimed_usd_for_user(user)
    if include_pending:
        try:
            from apps.wallets.models import PendingReward
            pending = PendingReward.objects.filter(user=user, withdrawn=False).select_related("token")
            pend_total = sum((_decimal(r.amount) * _decimal(_safe_getattr(r.token, "usd_price", 0))) for r in pending)
        except Exception:
            pend_total = Decimal("0")
        claimed += pend_total
    return (Decimal("1.0") + (claimed / principal)).quantize(Decimal("1.0000"))

def _top_roi_award(threshold_multiple: Decimal):
    b = Badge.objects.filter(is_active=True, type="TOP_ROI").first()
    if not b:
        return 0
    created = 0
    for u in User.objects.all().only("id"):
        roi = compute_user_roi_multiple(u)
        if roi >= threshold_multiple:
            if grant_badge(u, b):
                created += 1
    return created

def award_all_for_user(user: User) -> bool:
    before = user_badge_bonus_bp(user)
    _referral_thresholds_award_for_user(user)
    _level5_starter_award_for_user(user)
    _loyal_whale_award_for_user(user)
    after = user_badge_bonus_bp(user)
    return before != after

def award_top_roi_badge(threshold: float = 1.5) -> int:
    thr = Decimal(str(threshold))
    return _top_roi_award(thr)

def evaluate_badges_for_user(user: User) -> None:
    _first_n_users_award()
    award_all_for_user(user)
